<?php
// db/clients-update.php
ob_start(); // Iniciar el búfer de salida
error_reporting(0); // Desactivar la muestra de errores en producción
ini_set('display_errors', 0); // Desactivar la muestra de errores en producción

include_once 'db_connection.php'; // Incluir el archivo de conexión a la base de datos

$response = ['success' => false, 'message' => ''];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Error de JSON en clients-update.php: " . json_last_error_msg());
        $response['message'] = 'Error al decodificar datos de solicitud.';
        echo json_encode($response);
        exit();
    }

    $id = (int)($input['id'] ?? 0);
    $first_name = trim($input['first_name'] ?? '');
    $last_name = trim($input['last_name'] ?? '');
    $email = filter_var(trim($input['email'] ?? ''), FILTER_SANITIZE_EMAIL) ?: null;
    $phone = trim($input['phone'] ?? '') ?: null;
    $mobile = trim($input['mobile'] ?? '') ?: null;
    $city = trim($input['city'] ?? '') ?: null; // Se agregó la variable city
    $state_province = trim($input['estado'] ?? '') ?: null; // Usar 'estado' para state_province
    $assigned_service_ids = $input['assigned_service_ids'] ?? []; // Array de IDs de servicios

    if (empty($id) || empty($first_name) || empty($last_name)) {
        $response['message'] = 'ID de cliente, nombre y apellido son obligatorios.';
        echo json_encode($response);
        exit();
    }

    if ($email && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $response['message'] = 'El formato del correo electrónico no es válido.';
        echo json_encode($response);
        exit();
    }

    try {
        $pdo->beginTransaction();

        // Actualizar datos del cliente
        $stmt_client = $pdo->prepare("UPDATE clients SET first_name = :first_name, last_name = :last_name, email = :email, phone = :phone, mobile = :mobile, state_province = :state_province WHERE id = :id");
        $stmt_client->bindParam(':first_name', $first_name);
        $stmt_client->bindParam(':last_name', $last_name);
        $stmt_client->bindParam(':email', $email);
        $stmt_client->bindParam(':phone', $phone);
        $stmt_client->bindParam(':mobile', $mobile);
        $stmt_client->bindParam(':state_province', $state_province);
        $stmt_client->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt_client->execute();

        // Sincronizar servicios:
        // 1. Eliminar todos los servicios existentes para este cliente
        $stmt_delete_services = $pdo->prepare("DELETE FROM client_services WHERE client_id = :client_id");
        $stmt_delete_services->bindParam(':client_id', $id, PDO::PARAM_INT);
        $stmt_delete_services->execute();

        // 2. Insertar los servicios recién asignados
        if (!empty($assigned_service_ids)) {
            $stmt_insert_service = $pdo->prepare("INSERT INTO client_services (client_id, service_id) VALUES (:client_id, :service_id)");
            foreach ($assigned_service_ids as $service_id) {
                // Asegurarse de que el service_id sea un entero válido
                $service_id_int = (int)$service_id;
                if ($service_id_int > 0) { // Validar que sea un ID de servicio positivo
                    $stmt_insert_service->bindParam(':client_id', $id, PDO::PARAM_INT);
                    $stmt_insert_service->bindParam(':service_id', $service_id_int, PDO::PARAM_INT);
                    $stmt_insert_service->execute();
                }
            }
        }

        $pdo->commit();

        // Preparar la respuesta con los datos del cliente actualizado
        $response['success'] = true;
        $response['message'] = 'Cliente actualizado con éxito.';
        $response['client'] = [
            'id' => $id,
            'first_name' => $first_name,
            'last_name' => $last_name,
            'email' => $email,
            'phone' => $phone,
            'mobile' => $mobile,
            'city' => $city,
            'estado' => $state_province,
            'assigned_service_ids' => $assigned_service_ids // Devolver los IDs actualizados
        ];

    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Error PDO al actualizar cliente: " . $e->getMessage());
        $response['message'] = 'Error de Base de Datos al actualizar cliente: ' . $e->getMessage();
        // Si el error es por duplicado de email:
        if ($e->getCode() == 23000) {
            $response['message'] = 'El correo electrónico ya está registrado para otro cliente.';
        }
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Error general al actualizar cliente: " . $e->getMessage());
        $response['message'] = 'Error inesperado al actualizar cliente: ' . $e->getMessage();
    }
} else {
    $response['message'] = 'Método de solicitud no válido.';
}

ob_end_clean(); // Limpiar el búfer de salida
header('Content-Type: application/json'); // Establecer el tipo de contenido a JSON
echo json_encode($response); // Enviar la respuesta JSON
exit; // Terminar la ejecución del script