<?php
ob_start(); // Iniciar el búfer de salida al inicio del script
session_start();

// ***** IMPORTANTE: ESTAS LÍNEAS ESTÁN AQUÍ PARA DEPURAR. *****
// UNA VEZ QUE FUNCIONE, CÁMBIALAS A: error_reporting(0); ini_set('display_errors', 0);
error_reporting(E_ALL); // Reporta todos los errores
ini_set('display_errors', 1); // Muestra los errores en la pantalla
// *************************************************************

// Incluir el archivo de conexión a la base de datos
include_once 'db/db_connection.php';
include_once 'files/GuardianKey.php'; // Incluir si se usa para seguridad adicional

// --- Lógica PHP para obtener datos de servicios desde la base de datos ---
$servicios_data_from_db = [];
$total_services_count = 0;
$new_services_this_month = 0;
$categories_from_db = []; // Para almacenar las categorías leídas

try {
    // Construir la URL del endpoint de lectura de servicios
    $read_url_path = str_replace($_SERVER['DOCUMENT_ROOT'], '', realpath(__DIR__ . '/db/services-read.php'));
    $full_read_url = "http://" . $_SERVER['HTTP_HOST'] . $read_url_path;

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $full_read_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    $read_response = curl_exec($ch);

    if (curl_errno($ch)) {
        throw new Exception(curl_error($ch));
    }
    curl_close($ch);

    $result = json_decode($read_response, true);

    if ($result['success']) {
        $servicios_data_from_db = $result['data'];
        $total_services_count = $result['total_count'];
        $new_services_this_month = $result['new_this_month_count'];
        $categories_from_db = $result['categories'];
    } else {
        error_log("Error al cargar servicios desde db/services-read.php: " . ($result['message'] ?? 'Error desconocido'));
        $_SESSION['notification'] = ['message' => 'Error al cargar servicios: ' . ($result['message'] ?? 'Error desconocido'), 'type' => 'error'];
    }

} catch (Exception $e) {
    error_log("Excepción al cargar servicios en services.php: " . $e->getMessage());
    $_SESSION['notification'] = ['message' => 'Error crítico al cargar servicios: ' . $e->getMessage(), 'type' => 'error'];
}

// Codificar los datos a JSON para pasarlos a JavaScript
$servicios_json = json_encode($servicios_data_from_db);
$total_services_count_json = json_encode($total_services_count);
$new_services_this_month_json = json_encode($new_services_this_month);
$categories_json = json_encode($categories_from_db);


// Manejo de notificaciones de sesión
$notification_message = '';
$notification_type = '';
if (isset($_SESSION['notification'])) {
    $notification_message = $_SESSION['notification']['message'];
    $notification_type = $_SESSION['notification']['type'];
    unset($_SESSION['notification']);
}
ob_end_clean();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>LocalCRM ToolKit Dashboard | Administración de Servicios</title>
    <meta name="description" content="Administra tus servicios en LocalCRM: crea, edita y organiza tu catálogo, define precios y presenta opciones claras a tus clientes">
    <meta name="robots" content="noindex, nofollow">

    <link rel="icon" type="image/png" href="img/favicon.png">
    <link rel="apple-touch-icon" href="img/apple-touch-icon.png">


    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Barlow:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <script src="https://unpkg.com/lucide@latest"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    
    <link rel="stylesheet" href="style.css">
    <script src="files/header-manager.js"></script>
    </head>
    
<body data-page-title="PANEL DE ADMINISTRADOR - SERVICIOS"
      data-page-subtitle="GESTIÓN DE VERSIONES Y OFERTA CRM"
      data-page-icon="briefcase">

<div id="toastNotification" class="toast-notification">
    <i id="toastIcon" class="fas"></i>
    <span id="toastMessage"></span>
    <button id="toastCloseBtn" class="toast-close-btn">×</button>
</div>
<style>
.toast-notification {
    visibility: hidden;
    min-width: 250px;
    background-color: #333;
    color: #fff;
    text-align: center;
    border-radius: 8px;
    padding: 16px;
    position: fixed;
    z-index: 1000;
    left: 50%;
    transform: translateX(-50%);
    bottom: 30px;
    font-size: 17px;
    display: flex;
    align-items: center;
    gap: 10px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.2);
    opacity: 0;
    transition: opacity 0.5s, bottom 0.5s;
}

.toast-notification.show {
    visibility: visible;
    opacity: 1;
    bottom: 50px;
}

.toast-notification.success { background-color: #4CAF50; }
.toast-notification.error { background-color: #f44336; }
.toast-notification.info { background-color: #2196F3; }

.toast-notification .toast-close-btn {
    background: none;
    border: none;
    color: white;
    font-size: 20px;
    cursor: pointer;
    margin-left: auto;
}
</style>
    
<div class="relative min-h-screen md:flex">

    <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>

<?php include 'menu.php'; ?>

    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
            <button id="mobile-menu-button" class="md:hidden text-gray-600 hover:text-gray-800">
                <i data-lucide="menu" class="w-6 h-6"></i>
            </button>
            <div class="page-header-container">
                <h2 id="page-title"></h2>
                <p id="page-subtitle"></p>
            </div>
        </header>

        <div id="content-area" class="p-4 md:p-8 space-y-8">
            <section id="servicios" class="dashboard-section">
                <div class="bg-white p-6 rounded-xl shadow-md">
                    <div class="flex flex-col md:flex-row justify-between items-start md:items-center mb-6 gap-4">
                        <div>
                            <h3 class="text-2xl font-extrabold text-gray-800 flex items-center gap-2">
                                <i data-lucide="package" class="w-7 h-7 text-[var(--color-primary)]"></i> INVENTARIO DE SERVICIOS
                            </h3>
                            <p class="text-gray-500 text-sm mt-1 uppercase">VISUALIZA Y EDITA LAS VERSIONES Y SERVICIOS CRM.</p>
                        </div>
                        <div class="flex items-center gap-4 w-full md:w-auto">
                            <button class="btn-secondary font-bold py-2 px-4 rounded-lg flex items-center w-full md:w-auto justify-center uppercase" onclick="openPanel('addServicioPanel')">
                                <i data-lucide="plus" class="w-5 h-5 mr-2"></i> AGREGAR SERVICIO
                            </button>
                            <button class="btn-primary font-bold py-2 px-4 rounded-lg flex items-center w-full md:w-auto justify-center uppercase text-sm" onclick="openPanel('manageCategoriesPanel')">
                                <i data-lucide="tag" class="w-4 h-4 mr-2"></i> ADMINISTRAR CATEGORÍAS
                            </button>
                            <button id="download-servicios-btn" class="btn-primary font-bold py-2 px-4 rounded-lg flex items-center w-full md:w-auto justify-center uppercase text-sm">
                                <i data-lucide="download" class="w-4 h-4 mr-2"></i> DESCARGAR
                            </button>
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                        <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-primary)]">
                            <i data-lucide="package" class="w-12 h-12 text-gray-700"></i>
                            <div>
                                <h3 class="text-lg font-extrabold text-gray-500 mb-1">TOTAL DE SERVICIOS</h3>
                                <p id="total-services-count" class="text-5xl font-bold text-[var(--color-secondary)]">0</p>
                                <p class="text-sm text-gray-400 mt-1">EN EL CATÁLOGO</p>
                            </div>
                        </div>
                        <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-highlight)]">
                            <i data-lucide="wallet" class="w-12 h-12 text-gray-700"></i>
                            <div>
                                <h3 class="text-lg font-extrabold text-gray-500 mb-1">SERVICIOS NUEVOS ESTE MES</h3>
                                <p id="new-services-this-month-count" class="text-5xl font-bold text-[var(--color-secondary)]">0</p>
                                <p class="text-sm text-gray-400 mt-1 uppercase">AGREGADOS EN EL PERIODO</p>
                            </div>
                        </div>
                    </div>

                    <div class="flex flex-col md:flex-row gap-4 mb-6">
                        <div class="relative flex-grow">
                            <input type="text" id="service-search" placeholder="Buscar Por Nombre o Descripción..." class="w-full p-3 pl-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]">
                            <i data-lucide="search" class="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"></i>
                        </div>
                        <select id="service-filter-category" class="w-full md:w-48 p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white">
                            <option value="all">TODAS LAS CATEGORÍAS</option>
                            </select>
                        <div class="view-toggle inline-flex bg-gray-200 rounded-lg p-1">
                            <button id="grid-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 active uppercase" title="Vista de Cuadrícula">
                                <i data-lucide="layout-grid" class="w-5 h-5"></i>
                            </button>
                            <button id="list-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 uppercase" title="Vista de Lista">
                                <i data-lucide="list" class="w-5 h-5"></i>
                            </button>
                        </div>
                    </div>

                    <div id="service-grid-view" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    </div>

                    <div id="service-list-view" class="overflow-x-auto hidden">
                        <table class="min-w-full bg-white">
                            <thead class="bg-gray-50">
                                <tr class="text-left text-gray-500 uppercase text-sm">
                                    <th class="py-3 px-6 font-semibold">NOMBRE</th>
                                    <th class="py-3 px-6 font-semibold hidden md:table-cell">CATEGORÍA</th>
                                    <th class="py-3 px-6 font-semibold">PRECIO</th>
                                    <th class="py-3 px-6 font-semibold text-center">ACCIONES</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-700 text-sm" id="servicesTableBody">
                            </tbody>
                        </table>
                    </div>

                </div>
            </section>
        </div>
    </main>
</div>

<div id="addServicioPanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">AGREGAR NUEVO SERVICIO</h3>
            <button onclick="closePanel('addServicioPanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <form id="add-servicio-form" class="flex-grow overflow-y-auto pr-2 -mr-2">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-4">
                <div class="md:col-span-2">
                    <label for="servicio-nombre" class="block text-gray-700 text-sm font-bold mb-2 uppercase">NOMBRE DEL SERVICIO</label>
                    <input type="text" id="servicio-nombre" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. Limpieza Semanal" required>
                </div>
                <div>
                    <label for="servicio-precio" class="block text-gray-700 text-sm font-bold mb-2 uppercase">PRECIO</label>
                    <input type="number" step="0.01" id="servicio-precio" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. 50.00" required>
                </div>
                <div>
                    <label for="servicio-categoria" class="block text-gray-700 text-sm font-bold mb-2 uppercase">CATEGORÍA</label>
                    <select id="servicio-categoria" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white">
                        <option value="">Selecciona Una Categoría</option>
                        </select>
                </div>
                <div class="md:col-span-2">
                    <label for="servicio-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label>
                    <textarea id="servicio-descripcion" rows="3" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Descripción Del Servicio"></textarea>
                </div>
            </div>
            <div class="h-4"></div></form>
        <div class="flex justify-end space-x-4 pt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('addServicioPanel')">CANCELAR</button>
            <button type="submit" form="add-servicio-form" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase">GUARDAR SERVICIO</button>
        </div>
    </div>
</div>

<div id="confirmDeleteModal" class="fixed inset-0 bg-gray-900 bg-opacity-75 flex items-center justify-center hidden z-50">
    <div class="bg-white p-8 rounded-xl shadow-2xl w-full max-w-sm m-4 transform transition-all duration-300 scale-95 opacity-0 =">
        <div class="flex justify-center mb-4">
            <i data-lucide="alert-triangle" class="w-16 h-16 text-red-500"></i>
        </div>
        <h3 class="text-2xl font-bold text-[var(--color-primary)] mb-4 uppercase">CONFIRMAR ELIMINACIÓN</h3>
        <p class="text-gray-700 mb-6 uppercase">¿ESTÁS SEGURO DE QUE DESEAS ELIMINAR ESTE <span id="confirm-item-type" class="font-semibold">ELEMENTO</span>? ESTA ACCIÓN NO SE PUEDE DESHACER.</p>
        <div class="flex justify-center space-x-4">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closeModal('confirmDeleteModal')">CANCELAR</button>
            <button type="button" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase" id="confirm-delete-button">CONFIRMAR</button>
        </div>
    </div>
</div>

<div id="viewServicioPanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">DETALLE DEL SERVICIO</h3>
            <button onclick="closePanel('viewServicioPanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <form id="view-servicio-form" class="flex-grow overflow-y-auto pr-2 -mr-2">
            <input type="hidden" id="view-servicio-id">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-4">
                <div class="md:col-span-2">
                    <label for="view-servicio-nombre" class="block text-gray-700 text-sm font-bold mb-2 uppercase">NOMBRE DEL SERVICIO</label>
                    <input type="text" id="view-servicio-nombre" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Nombre Del servicio">
                </div>
                <div>
                    <label for="view-servicio-precio" class="block text-gray-700 text-sm font-bold mb-2 uppercase">PRECIO</label>
                    <input type="number" step="0.01" id="view-servicio-precio" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. 50.00">
                </div>
                <div>
                    <label for="view-servicio-categoria" class="block text-gray-700 text-sm font-bold mb-2 uppercase">CATEGORÍA</label>
                    <select id="view-servicio-categoria" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white">
                        <option value="">Selecciona Una Categoría</option>
                        </select>
                </div>
                <div class="md:col-span-2">
                    <label for="view-servicio-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label>
                    <textarea id="view-servicio-descripcion" rows="3" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Descripción Del Servicio"></textarea>
                </div>
            </div>
            <div class="h-4"></div></form>
        <div class="flex justify-end space-x-4 pt-4 border-t border-gray-200 mt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('viewServicioPanel')">CANCELAR</button>
            <button type="submit" form="view-servicio-form" class="btn-primary font-bold py-2 px-4 rounded-lg uppercase">GUARDAR CAMBIOS</button>
        </div>
    </div>
</div>

<div id="manageCategoriesPanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">ADMINISTRAR CATEGORÍAS</h3>
            <button onclick="closePanel('manageCategoriesPanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <div class="flex-grow overflow-y-auto pr-2 -mr-2">
            <h4 class="text-xl font-bold text-gray-700 mb-4 uppercase">Agregar Nueva Categoría</h4>
            <form id="add-category-form" class="mb-8">
                <div class="mb-4">
                    <label for="new-category-name" class="block text-gray-700 text-sm font-bold mb-2 uppercase">Nombre De La Categoría</label>
                    <input type="text" id="new-category-name" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. Contabilidad" required>
                </div>
                <button type="submit" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase">AÑADIR CATEGORÍA</button>
            </form>

            <h4 class="text-xl font-bold text-gray-700 mb-4 uppercase">Categorías Existentes</h4>
            <div id="category-list-container" class="space-y-2">
                <p class="text-gray-500 text-sm italic">Cargando Categorías...</p>
            </div>
        </div>
        <div class="flex justify-end space-x-4 pt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('manageCategoriesPanel')">CERRAR</button>
        </div>
    </div>
</div>


<script>
// Variable global para almacenar los servicios cargados desde PHP
let serviciosActuales = (function() {
    try {
        const data = <?php echo $servicios_json; ?>;
        return (typeof data === 'object' && data !== null && !Array.isArray(data)) ? data : {};
    } catch (e) {
        console.error("Error al parsear servicios_json:", e);
        return {};
    }
})();
// Variable para almacenar la lista de categorías
let allCategories = (function() {
    try {
        const data = <?php echo $categories_json; ?>;
        return (Array.isArray(data)) ? data : [];
    } catch (e) {
        console.error("Error al parsear categories_json:", e);
        return [];
    }
})();
// Variable para almacenar el contador de servicios nuevos este mes
let newServicesThisMonthCount = <?php echo $new_services_this_month_json; ?>;


document.addEventListener('DOMContentLoaded', function() {
    // --- CONSOLE.LOGS PARA DEPURACIÓN ---
    console.log("--- Depuración de services.php ---");
    console.log("serviciosActuales (datos de servicios de la DB al cargar PHP):", serviciosActuales);
    console.log("newServicesThisMonthCount (contador mensual desde PHP):", newServicesThisMonthCount);
    console.log("allCategories (categorías desde PHP):", allCategories);
    // --- FIN CONSOLE.LOGS PARA DEPURACIÓN ---


    // --- VARIABLES DOM LOCALES ---
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const sidebar = document.getElementById('sidebar');
    const sidebarOverlay = document.getElementById('sidebar-overlay');
    const serviceSearch = document.getElementById('service-search');
    const serviceFilterCategory = document.getElementById('service-filter-category');
    const serviceGridView = document.getElementById('service-grid-view');
    const serviceListView = document.getElementById('service-list-view');
    const servicesTableBody = document.getElementById('servicesTableBody');
    const gridViewBtn = document.getElementById('grid-view-btn');
    const listViewBtn = document.getElementById('list-view-btn');
    const downloadServiciosBtn = document.getElementById('download-servicios-btn');
    const confirmDeleteButton = document.getElementById('confirm-delete-button');
    const confirmItemTypeSpan = document.getElementById('confirm-item-type');
    const addServicioForm = document.getElementById('add-servicio-form');
    const viewServicioForm = document.getElementById('view-servicio-form');
    const totalServicesCountElement = document.getElementById('total-services-count');
    const newServicesThisMonthCountElement = document.getElementById('new-services-this-month-count');
    // NUEVAS referencias DOM para categorías
    const addServiceCategorySelect = document.getElementById('servicio-categoria');
    const viewServiceCategorySelect = document.getElementById('view-servicio-categoria');
    const newCategoryNameInput = document.getElementById('new-category-name');
    const addCategoryForm = document.getElementById('add-category-form');
    const categoryListContainer = document.getElementById('category-list-container');


    let currentServiceView = 'grid';
    
    // Elemento de overlay para los paneles laterales (creado dinámicamente)
    const panelOverlay = document.createElement('div');
    panelOverlay.id = 'panel-overlay';
    panelOverlay.className = 'fixed inset-0 bg-black bg-opacity-50 z-40 hidden';
    document.body.appendChild(panelOverlay);


    // --- FUNCIONES HELPER GLOBALES ---

    // Muestra una notificación personalizada
    window.showCustomNotification = function(message, type = 'info', duration = 3000) {
        const toast = document.getElementById('toastNotification');
        const toastMessage = document.getElementById('toastMessage');
        const toastIcon = document.getElementById('toastIcon');
        const toastCloseBtn = document.getElementById('toastCloseBtn');
        let toastTimeout;

        if (!toast || !toastMessage || !toastIcon || !toastCloseBtn) {
            console.warn(`[NOTIFICACION - FALLBACK] Elementos de toast no encontrados. Mensaje: ${type.toUpperCase()}: ${message}`);
            return;
        }

        clearTimeout(toastTimeout);
        toast.className = 'toast-notification'; // Resetear clases
        toastIcon.className = 'fas'; // Resetear icono base

        toastMessage.textContent = message;
        toast.classList.add(type === 'success' ? 'success' : (type === 'error' ? 'error' : 'info'));
        
        if (type === 'success') {
            toastIcon.classList.add('fa-check-circle');
        } else if (type === 'error') {
            toastIcon.classList.add('fa-exclamation-circle');
        } else {
            toastIcon.classList.add('fa-info-circle');
        }

        toast.classList.add('show');
        toastTimeout = setTimeout(() => {
            toast.classList.remove('show');
        }, duration);

        toastCloseBtn.onclick = () => {
            clearTimeout(toastTimeout);
            toast.classList.remove('show');
        };

        console.log(`[NOTIFICACION - ${type.toUpperCase()}] ${message}`);
    };

    // Abre un panel lateral
    window.openPanel = function(panelId) {
        const panel = document.getElementById(panelId);
        if (!panel) { console.error(`Error: Panel con ID "${panelId}" no encontrado.`); return; }
        panel.classList.remove('translate-x-full');
        panelOverlay.classList.remove('hidden');
        console.log(`[services.php] Panel ${panelId} abierto.`);
        
        // Lógica específica para el panel de agregar servicio (si aplica, p.ej. resetear formulario)
        if (panelId === 'addServicioPanel') {
            addServicioForm.reset();
            populateCategoryDropdowns(); // Llenar el dropdown de categorías al abrir
        } else if (panelId === 'manageCategoriesPanel') {
            newCategoryNameInput.value = ''; // Limpiar input al abrir
            renderCategoryList(); // Renderizar la lista de categorías existentes
        }
    };

    // Cierra un panel lateral
    window.closePanel = function(panelId) {
        const panel = document.getElementById(panelId);
        if (!panel) { console.error(`Error: Panel con ID "${panelId}" no encontrado para cerrar.`); return; }
        panel.classList.add('translate-x-full');
        panelOverlay.classList.add('hidden');
        console.log(`[services.php] Panel ${panelId} cerrado.`);
    };

    // Abre un modal (solo para confirmDeleteModal)
    window.openModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) { console.error(`Error: Modal con ID "${modalId}" no encontrado.`); return; }
        const modalBox = modal.querySelector('div:first-of-type');
        modal.classList.remove('hidden');
        setTimeout(() => { modalBox.classList.remove('scale-95', 'opacity-0'); }, 50);
        console.log(`[services.php] Modal ${modalId} abierto.`);
    };

    // Cierra un modal (solo para confirmDeleteModal)
    window.closeModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) { console.error(`Error: Modal con ID "${modalId}" no encontrado para cerrar.`); return; }
        const modalBox = modal.querySelector('div:first-of-type');
        modalBox.classList.add('scale-95', 'opacity-0');
        setTimeout(() => { modal.classList.add('hidden'); }, 300);
        console.log(`[services.php] Modal ${modalId} cerrado.`);
    };
    
    // Abre el modal de confirmación de eliminación
    window.openConfirmDeleteModal = function(itemId, itemType) {
        if (!confirmDeleteButton || !confirmItemTypeSpan) { console.error("Error: Elementos del modal de confirmación de eliminación no encontrados."); return; }
        confirmDeleteButton.dataset.itemId = itemId; confirmDeleteButton.dataset.itemType = itemType;
        confirmItemTypeSpan.textContent = itemType.toUpperCase();
        openModal('confirmDeleteModal');
    };

    // Descarga datos como CSV
    function descargarComoCSV(dataObject, filename) {
        const data = Object.values(dataObject); // Convertir objeto a array de valores
        if (data.length === 0) { showCustomNotification('No hay datos para descargar.', 'info'); return; }
        const escapeCSV = (value) => {
            if (value === null || value === undefined) { return ''; }
            value = String(value);
            if (value.search(/("|,|\n)/g) >= 0) { return `"${value.replace(/"/g, '""')}"`; }
            return value;
        };
        // Asegúrate de que los encabezados correspondan a las propiedades de tus objetos de servicio
        const headers = ['db_id', 'nombre', 'precio', 'categoria', 'descripcion'];
        const csvRows = [headers.map(h => escapeCSV(h.replace(/_/g, ' ').toUpperCase())).join(',')];
        data.forEach(item => {
            const values = headers.map(header => escapeCSV(item[header]));
            csvRows.push(values.join(','));
        });
        const csvContent = csvRows.join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        if (link.download !== undefined) {
            const url = URL.createObjectURL(blob);
            link.setAttribute('href', url); link.setAttribute('download', filename);
            link.style.visibility = 'hidden'; document.body.appendChild(link);
            link.click(); document.body.removeChild(link);
        }
    }

    // Abre el panel de Ver/Editar Servicio
    window.openViewServicioModal = function(servicioId) {
        const servicio = serviciosActuales[servicioId]; // Acceso directo por clave de string
        if (!servicio) { 
            console.error('Servicio no encontrado:', servicioId); 
            showCustomNotification('Error: Servicio no encontrado para editar.', 'error');
            return; 
        }
        document.getElementById('view-servicio-id').value = servicio.db_id; // db_id es el id numérico de la DB
        document.getElementById('view-servicio-nombre').value = servicio.nombre;
        document.getElementById('view-servicio-precio').value = servicio.precio;
        // Llenar y seleccionar en el dropdown de edición
        populateCategoryDropdowns(servicio.categoria); // Pasa el nombre de la categoría
        document.getElementById('view-servicio-descripcion').value = servicio.descripcion || '';
        openPanel('viewServicioPanel'); 
    };

    // Setup de listeners (sin cambios en la lógica interna)
    function setupDeleteListeners() {
        document.querySelectorAll('.delete-btn').forEach(button => {
            button.removeEventListener('click', handleDeleteButtonClick);
            button.addEventListener('click', handleDeleteButtonClick);
        });
    }
    function handleDeleteButtonClick(e) {
        e.stopPropagation();
        const itemId = this.dataset.itemId; const itemType = this.dataset.itemType;
        openConfirmDeleteModal(itemId, itemType);
    }

    function setupEditListeners() {
        document.querySelectorAll('.edit-servicio-btn').forEach(button => {
            button.removeEventListener('click', handleEditServicioButtonClick);
            button.addEventListener('click', handleEditServicioButtonClick);
        });
    }
    function handleEditServicioButtonClick(e) {
        e.stopPropagation();
        const servicioId = this.dataset.servicioId;
        openViewServicioModal(servicioId);
    }
    
    function toggleSidebar() {
        if (sidebar) sidebar.classList.toggle('-translate-x-full');
        if (sidebarOverlay) sidebar.classList.toggle('hidden');
    }

    function updateTotalServicesDisplay() {
        if (totalServicesCountElement) {
            const total = Object.keys(serviciosActuales).length; // Contar las claves del objeto
            totalServicesCountElement.textContent = total.toString();
        }
    }

    // Función para actualizar el contador de servicios nuevos este mes
    function updateNewServicesThisMonthDisplay() {
        if (newServicesThisMonthCountElement) {
            let newThisMonth = 0;
            const currentMonth = new Date().getMonth();
            const currentYear = new Date().getFullYear();
            Object.values(serviciosActuales).forEach(service => {
                if (service.created_at) {
                    const createdAt = new Date(service.created_at);
                    if (createdAt.getFullYear() === currentYear && createdAt.getMonth() === currentMonth) {
                        newThisMonth++;
                    }
                }
            });
            newServicesThisMonthCountElement.textContent = newThisMonth.toString();
        }
    }


    // --- Función para cargar los servicios y categorías desde la DB ---
    const fetchServicesAndCategories = async () => {
        try {
            const response = await fetch('db/services-read.php?' + new Date().getTime()); // Añadir timestamp para evitar caché
            const result = await response.json();

            if (result.success && result.data) {
                serviciosActuales = result.data;
                allCategories = result.categories; // Asignar categorías del backend
                console.log("Servicios cargados desde DB:", serviciosActuales);
                console.log("Categorías cargadas desde DB:", allCategories);

                updateTotalServicesDisplay();
                updateNewServicesThisMonthDisplay();
                populateCategoryFilter(); // Poblar el filtro principal
                populateCategoryDropdowns(); // Poblar los dropdowns de formularios
                renderCategoryList(); // Renderizar lista de categorías en panel
                renderServices(currentServiceView);
            } else {
                showCustomNotification(result.message || 'Error al cargar los servicios desde la base de datos.', 'error');
                serviciosActuales = {}; // Vaciar datos si hay error
                allCategories = []; // Vaciar categorías
                updateTotalServicesDisplay();
                updateNewServicesThisMonthDisplay();
                renderServices(currentServiceView);
            }
        } catch (error) {
            showCustomNotification('Error de conexión al cargar los servicios.', 'error');
            console.error('Error fetching services:', error);
            serviciosActuales = {}; // Vaciar datos si hay error
            allCategories = []; // Vaciar categorías
            updateTotalServicesDisplay();
            updateNewServicesThisMonthDisplay();
            renderServices(currentServiceView);
        }
    };


    // Función para poblar el filtro principal de categorías
    function populateCategoryFilter() {
        if (!serviceFilterCategory) return;
        const filterCategories = new Set(allCategories.map(cat => cat.name)); 
        const sortedCategories = Array.from(filterCategories).sort();
        serviceFilterCategory.innerHTML = '<option value="all">TODAS LAS CATEGORÍAS</option>';
        sortedCategories.forEach(categoryName => {
            const option = document.createElement('option');
            option.value = categoryName;
            option.textContent = categoryName.toUpperCase();
            serviceFilterCategory.appendChild(option);
        });
    }

    // Función para poblar los dropdowns de categoría en los formularios de añadir/editar
    function populateCategoryDropdowns(selectedCategoryName = '') {
        if (!addServiceCategorySelect || !viewServiceCategorySelect) return;

        // Limpiar opciones existentes
        addServiceCategorySelect.innerHTML = '<option value="">Selecciona una categoría</option>';
        viewServiceCategorySelect.innerHTML = '<option value="">Selecciona una categoría</option>';

        const sortedCategories = [...allCategories].sort((a, b) => a.name.localeCompare(b.name));

        sortedCategories.forEach(category => {
            const optionAdd = document.createElement('option');
            optionAdd.value = category.name; // Usar el nombre de la categoría como valor
            optionAdd.textContent = category.name;
            if (category.name === selectedCategoryName) {
                optionAdd.selected = true;
            }
            addServiceCategorySelect.appendChild(optionAdd);

            const optionView = document.createElement('option');
            optionView.value = category.name; // Usar el nombre de la categoría como valor
            optionView.textContent = category.name;
            if (category.name === selectedCategoryName) {
                optionView.selected = true;
            }
            viewServiceCategorySelect.appendChild(optionView);
        });
    }

    // Función para renderizar la lista de categorías en el panel de administración
    function renderCategoryList() {
        if (!categoryListContainer) return;
        categoryListContainer.innerHTML = ''; 
        if (allCategories.length === 0) {
            categoryListContainer.innerHTML = '<p class="text-gray-500 text-sm italic">No Hay Categorías Definidas. Añade Una.</p>';
            return;
        }

        const sortedCategories = [...allCategories].sort((a, b) => a.name.localeCompare(b.name));
        sortedCategories.forEach(category => {
            const categoryItem = document.createElement('div');
            // Usamos category.name para el data-category-name del botón de eliminar (ya que el backend no tiene un ID real para categorías)
            categoryItem.className = 'flex items-center justify-between p-3 bg-gray-100 rounded-lg border border-gray-200';
            categoryItem.innerHTML = `
                <span class="text-gray-800 font-semibold uppercase">${category.name}</span>
                <button type="button" class="text-red-500 hover:text-red-700 delete-category-btn" data-category-name="${category.name}" title="Eliminar categoría">
                    <i data-lucide="trash-2" class="w-5 h-5"></i>
                </button>
            `;
            categoryListContainer.appendChild(categoryItem);
        });
        lucide.createIcons({ container: categoryListContainer }); 

        // Adjuntar listeners para botones de eliminar categoría
        categoryListContainer.querySelectorAll('.delete-category-btn').forEach(button => {
            button.removeEventListener('click', handleDeleteCategoryClick); // Evitar duplicados
            button.addEventListener('click', handleDeleteCategoryClick);
        });
    }

    // Manejador para el botón de eliminar categoría (SIMULADO - NO HAY ENDPOINT DEDICADO)
    async function handleDeleteCategoryClick() {
        const categoryNameToDelete = this.dataset.categoryName; 
        
        // Validar si la categoría está en uso por algún servicio
        const isCategoryInUse = Object.values(serviciosActuales).some(service => service.categoria === categoryNameToDelete);
        
        if (isCategoryInUse) {
            showCustomNotification('No se puede eliminar la categoría porque está asignada a uno o más servicios.', 'error', 5000);
            return;
        }

        openConfirmDeleteModal(categoryNameToDelete, 'categoría'); // Reutiliza el modal de confirmación

        confirmDeleteButton.onclick = async () => {
            showCustomNotification(`Eliminando categoría (simulado)...`, 'info');
            try {
                // SIMULACIÓN DE ELIMINACIÓN DE CATEGORÍA LOCAL
                const initialCategoriesLength = allCategories.length;
                allCategories = allCategories.filter(cat => cat.name !== categoryNameToDelete);
                
                if (allCategories.length < initialCategoriesLength) {
                    showCustomNotification('Categoría eliminada con éxito (simulado).', 'success');
                    // Re-renderizar todo lo que dependa de las categorías
                    populateCategoryFilter();
                    populateCategoryDropdowns();
                    renderCategoryList();
                    renderServices(currentServiceView); // Re-renderizar servicios por si había un servicio en esa categoría
                } else {
                    showCustomNotification('Error al eliminar categoría (simulado): Categoría no encontrada.', 'error');
                }
            } catch (error) {
                showCustomNotification('Error de conexión al eliminar la categoría (simulado).', 'error');
                console.error('Delete category error (simulado):', error);
            }
            closeModal('confirmDeleteModal');
        };
    }


    function renderServiceCard(service) {
        const card = document.createElement('div');
        card.className = 'bg-white rounded-xl shadow-md overflow-hidden flex flex-col border border-gray-200';
        card.dataset.servicioId = service.id; // Usar el ID de string
        card.dataset.itemId = service.id; // Usar el ID de string
        card.innerHTML = `
            <div class="bg-[var(--color-primary)] text-white p-4 flex items-center justify-between">
                <div class="flex items-center gap-2 flex-grow min-w-0">
                    <i data-lucide="clipboard-check" class="w-5 h-5 flex-shrink-0"></i>
                    <h3 class="font-bold text-lg uppercase truncate">${service.nombre}</h3>
                </div>
                </div>
            <div class="p-6 flex-grow flex flex-col">
                <p class="text-gray-600 uppercase text-sm mb-2 font-semibold">PRECIO DEL SERVICIO:</p>
                <p class="text-[var(--color-secondary)] font-extrabold text-3xl mb-4">$${parseFloat(service.precio).toFixed(2)}</p>
                <p class="text-gray-700 text-sm flex-grow mb-4 line-clamp-3">${service.descripcion || 'Sin descripción detallada.'}</p>
                <p class="text-gray-500 text-xs uppercase mt-auto">Categoría: <span class="font-semibold">${service.categoria || 'N/A'}</span></p>
            </div>
            <div class="bg-gray-100 p-3 flex justify-end space-x-3 border-t border-gray-200">
                <button class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-semibold py-2 px-4 rounded-lg text-sm uppercase edit-servicio-btn" data-servicio-id="${service.id}">EDITAR</button>
                <button class="p-2 bg-red-100 hover:bg-red-200 text-red-600 rounded-lg delete-btn" data-item-type="servicio" data-item-id="${service.db_id}" title="ELIMINAR SERVICIO"><i data-lucide="trash-2" class="w-5 h-5"></i></button>
            </div>
        `;
        return card;
    }

    function renderServiceTableRow(service) {
        const row = document.createElement('tr');
        row.className = 'border-b border-gray-200 hover:bg-gray-50';
        row.dataset.servicioId = service.id; // Usar el ID de string
        row.dataset.itemId = service.id; // Usar el ID de string
        row.innerHTML = `
            <td class="py-4 px-6 whitespace-nowrap">
                <span class="text-blue-600 hover:text-blue-800 cursor-pointer font-semibold uppercase" onclick="openViewServicioModal('${service.id}')">${service.nombre}</span>
            </td>
            <td class="py-4 px-6 hidden md:table-cell uppercase">${service.categoria || 'N/A'}</td>
            <td class="py-4 px-6 uppercase">$${parseFloat(service.precio).toFixed(2)}</td>
            <td class="py-4 px-6 text-center">
                <button class="text-blue-600 hover:text-blue-800 mr-3 edit-servicio-btn" title="EDITAR SERVICIO" data-servicio-id="${service.id}"><i data-lucide="edit-2" class="w-5 h-5"></i></button>
                <button class="text-red-600 hover:text-red-800 delete-btn" data-item-type="servicio" data-item-id="${service.db_id}" title="ELIMINAR SERVICIO"><i data-lucide="trash-2" class="w-5 h-5"></i></button>
            </td>
        `;
        return row;
    }

    function renderServices(viewType = currentServiceView) {
        if (!serviceGridView || !serviceListView || !servicesTableBody || !serviceSearch || !serviceFilterCategory || !gridViewBtn || !listViewBtn) {
                console.error("[services.php] ERROR: Elementos DOM de Servicios no encontrados. Verifique los IDs de los elementos HTML."); return;
        }
        const searchTerm = serviceSearch.value.toLowerCase();
        const selectedCategory = serviceFilterCategory.value;
        const allServicesArray = Object.values(serviciosActuales); // Convertir el objeto a array para filtrar

        console.log("renderServices: serviciosActuales antes de filtrar:", allServicesArray);

        const filteredServices = allServicesArray.filter(s => {
            const matchesSearch = searchTerm === '' || s.nombre.toLowerCase().includes(searchTerm) || (s.descripcion && s.descripcion.toLowerCase().includes(searchTerm));
            const matchesCategory = selectedCategory === 'all' || (s.categoria && s.categoria.toLowerCase().includes(selectedCategory.toLowerCase()));
            return matchesSearch && matchesCategory;
        });

        console.log("renderServices: servicios filtrados:", filteredServices);


        serviceGridView.innerHTML = '';
        servicesTableBody.innerHTML = '';

        if (filteredServices.length === 0) {
            const message = `<p class="text-gray-500 md:col-span-2 lg:col-span-3 text-center uppercase">NO SE ENCONTRARON SERVICIOS.</p>`;
            if (viewType === 'grid') {
                serviceGridView.innerHTML = message;
            } else {
                servicesTableBody.innerHTML = `<tr><td colspan="4" class="py-4 px-6 text-center text-gray-500 uppercase">NO SE ENCONTRARON SERVICIOS.</td></tr>`;
            }
        } else {
            if (viewType === 'grid') {
                filteredServices.forEach(service => serviceGridView.appendChild(renderServiceCard(service)));
            } else {
                filteredServices.forEach(service => servicesTableBody.appendChild(renderServiceTableRow(service)));
            }
        }

        if (viewType === 'grid') {
            serviceGridView.classList.remove('hidden');
            serviceListView.classList.add('hidden');
            gridViewBtn.classList.add('active');
            listViewBtn.classList.remove('active');
        } else {
            serviceGridView.classList.add('hidden');
            serviceListView.classList.remove('hidden');
            gridViewBtn.classList.remove('active');
            listViewBtn.classList.add('active');
        }

        setupDeleteListeners();
        setupEditListeners();
        lucide.createIcons();
    }
    
    // Funciones de formateo de teléfono (se mantienen aunque no se usan directamente en esta página)
    function formatPhoneNumber(value) {
        if (!value) return '';
        const cleaned = ('' + value).replace(/\D/g, '');    
        const match = cleaned.match(/^(\d{3})(\d{3})(\d{4})$/);
        if (match) {
            return `(${match[1]}) ${match[2]}-${match[3]}`;
        }
        return cleaned;    
    }

    function applyInputPhoneFormat(inputElement) {
        if (!inputElement) return;
        inputElement.value = formatPhoneNumber(inputElement.value);
        inputElement.addEventListener('input', (e) => {
            let value = e.target.value.replace(/\D/g, '');    
            e.target.value = formatPhoneNumber(value);
        });
        inputElement.addEventListener('blur', (e) => {
            e.target.value = formatPhoneNumber(e.target.value);
        });
    }

    function applyPhoneFormatToInputs() {
        // No hay campos de teléfono en esta página de servicios.
    }


    // --- INICIALIZACIÓN DE LA PÁGINA ---

    // Mostrar notificación PHP si existe
    const phpNotificationMessage = "<?php echo addslashes($notification_message); ?>";
    const phpNotificationType = "<?php echo addslashes($notification_type); ?>";
    if (phpNotificationMessage) {
        showCustomNotification(phpNotificationMessage, phpNotificationType);
    }

    // Eventos de la barra lateral (menú móvil)
    const sidebarCloseBtn = document.getElementById('sidebarCloseBtn');
    if (mobileMenuButton) mobileMenuButton.addEventListener('click', toggleSidebar);
    // Cierra los paneles y el overlay si se hace clic en el overlay de la barra lateral
    if (sidebarOverlay) sidebarOverlay.addEventListener('click', () => {
        toggleSidebar();
        closePanel('addServicioPanel');
        closePanel('viewServicioPanel');
        closePanel('manageCategoriesPanel');
    });
    // Este listener en panelOverlay ahora es crucial para cerrar los paneles laterales
    panelOverlay.addEventListener('click', () => {    
        closePanel('addServicioPanel');
        closePanel('viewServicioPanel');
        closePanel('manageCategoriesPanel'); 
    });
    if (sidebarCloseBtn) sidebarCloseBtn.addEventListener('click', toggleSidebar);    

    // --- Carga inicial de servicios y categorías ---
    fetchServicesAndCategories(); // Llama a la función para cargar datos de la DB


    // Renderizar servicios en la vista inicial (grid por defecto)
    // Esto se llamará dentro de fetchServicesAndCategories
    // renderServices(currentServiceView); 
    console.log("[services.php] Servicios renderizados.");

    if (serviceSearch) serviceSearch.addEventListener('input', () => renderServices(currentServiceView));
    if (serviceFilterCategory) serviceFilterCategory.addEventListener('change', () => renderServices(currentServiceView));

    if(downloadServiciosBtn) {
        downloadServiciosBtn.addEventListener('click', () => {
            descargarComoCSV(serviciosActuales, 'catalogo_servicios.csv');
        });
    }
    
    // Lógica para el botón de confirmación de eliminación (para servicios y categorías)
    if (confirmDeleteButton) {
        confirmDeleteButton.addEventListener('click', async function() {
            const itemId = this.dataset.itemId;
            const itemType = this.dataset.itemType; // 'servicio' o 'categoría'

            if (itemType === 'servicio') {
                try {
                    const response = await fetch('db/services-delete.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ id: itemId }) // itemId aquí ya es el db_id numérico
                    });
                    const result = await response.json();

                    if (result.success) {
                        closeModal('confirmDeleteModal');
                        showCustomNotification(result.message, 'success');
                        fetchServicesAndCategories(); // Recargar datos
                    } else {
                        showCustomNotification(result.message || 'Error al eliminar servicio.', 'error');
                    }
                } catch (error) {
                    showCustomNotification('Error de conexión al eliminar el servicio.', 'error');
                    console.error('Delete service error:', error);
                }
            } else if (itemType === 'categoría') {
                // Ya se maneja la eliminación de categorías localmente y con validación antes de confirmar
                // Si el backend tuviera un endpoint para eliminar categorías, se haría aquí.
                showCustomNotification('La eliminación de categorías se maneja solo a nivel visual si no está en uso. Recargue la página para revertir cambios no persistidos.', 'info', 5000);
                closeModal('confirmDeleteModal');
            }
        });
    }

    if (gridViewBtn) {
        gridViewBtn.addEventListener('click', () => {
            currentServiceView = 'grid';
            renderServices('grid');
        });
    }
    if (listViewBtn) {
        listViewBtn.addEventListener('click', () => {
            currentServiceView = 'list';
            renderServices('list');
        });
    }

    if (addServicioForm) {
        addServicioForm.addEventListener('submit', async function(e) {    
            e.preventDefault();
            const serviceData = {
                name: document.getElementById('servicio-nombre').value,
                price: parseFloat(document.getElementById('servicio-precio').value),
                category: addServiceCategorySelect.value,    
                description: document.getElementById('servicio-descripcion').value
            };

            try {    
                const response = await fetch('db/services-create.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(serviceData)
                });
                const result = await response.json();

                if (result.success) {
                    closePanel('addServicioPanel');
                    addServicioForm.reset();
                    showCustomNotification(result.message, 'success');
                    fetchServicesAndCategories(); // Recargar datos y categorías
                } else {
                    showCustomNotification(result.message || 'Error al agregar servicio.', 'error');
                }
            } catch (error) {
                showCustomNotification('Error de conexión al agregar el servicio.', 'error');
                console.error('Create service error:', error);
            }
        });
    }

    if (viewServicioForm) {
        viewServicioForm.addEventListener('submit', async function(e) {    
            e.preventDefault();
            const serviceData = {
                id: document.getElementById('view-servicio-id').value, // Esto es el db_id
                name: document.getElementById('view-servicio-nombre').value,
                price: parseFloat(document.getElementById('view-servicio-precio').value),
                category: viewServiceCategorySelect.value,    
                description: document.getElementById('view-servicio-descripcion').value
            };
            
            try {    
                const response = await fetch('db/services-update.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(serviceData)
                });
                const result = await response.json();

                if (result.success) {
                    closePanel('viewServicioPanel');
                    showCustomNotification(result.message, 'success');
                    fetchServicesAndCategories(); // Recargar datos y categorías
                } else {
                    showCustomNotification(result.message || 'Error al guardar cambios.', 'error');
                }
            } catch (error) {
                showCustomNotification('Error de conexión al guardar los cambios.', 'error');
                console.error('Update service error:', error);
            }
        });
    }

    // --- Lógica para el panel de administración de categorías (simulada) ---
    if (addCategoryForm) {
        addCategoryForm.addEventListener('submit', async function(e) {    
            e.preventDefault();
            const newCategoryName = newCategoryNameInput.value.trim();
            if (newCategoryName) {
                // Verificar si la categoría ya existe localmente (sensible a mayúsculas/minúsculas)
                const categoryExists = allCategories.some(cat => cat.name === newCategoryName);
                if (categoryExists) {
                    showCustomNotification('Esta categoría ya existe.', 'warning');
                    return;
                }
                
                showCustomNotification(`Añadiendo categoría (simulado)...`, 'info');
                try {
                    // SIMULACIÓN DE CREACIÓN DE CATEGORÍA LOCAL
                    // Generar un ID simple para la representación local. NO se guarda en DB.
                    const newCategoryId = allCategories.length > 0 ? Math.max(...allCategories.map(c => c.id)) + 1 : 1;
                    
                    const simulatedNewCategory = { id: newCategoryId, name: newCategoryName };
                    
                    allCategories.push(simulatedNewCategory);
                    allCategories.sort((a,b) => a.name.localeCompare(b.name)); // Re-ordenar
                    
                    showCustomNotification('Categoría añadida con éxito (simulado).', 'success');
                    newCategoryNameInput.value = ''; // Limpiar campo
                    
                    // Re-renderizar todo lo que dependa de las categorías
                    populateCategoryFilter();
                    populateCategoryDropdowns();
                    renderCategoryList();

                    // NOTA: Si necesita persistencia de categorías en DB, deberá crear una tabla `categories`
                    // y un endpoint `db/categories-create.php`. Por ahora, solo es visual/local.
                } catch (error) {
                    showCustomNotification('Error inesperado al añadir la categoría (simulado).', 'error');
                    console.error('Create category error (simulado):', error);
                }
            } else {
                showCustomNotification('El nombre de la categoría no puede estar vacío.', 'error');
            }
        });
    }

    // Renderiza todos los íconos de Lucide estáticos al final de la carga del DOM
    lucide.createIcons();
});
</script>
</body>
</html>