<?php
header('Content-Type: application/json');

// Incluye el archivo de configuración con la API Key.
// La ruta 'config.php' significa:
// "Desde la ubicación actual de este archivo, busca 'config.php' en la misma carpeta."
require_once 'config.php';

// Ahora, la API Key se toma de la constante definida en config.php
$apiKey = GEMINI_API_KEY;

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido. Utiliza POST.']);
    exit;
}

// Obtener y decodificar los datos JSON de la solicitud
$json_data = file_get_contents('php://input');
$data = json_decode($json_data, true);

// Validar que los datos requeridos estén presentes
if (!$data || !isset($data['ad_text_es']) || empty(trim($data['ad_text_es']))) {
    http_response_code(400);
    echo json_encode(['error' => 'No se recibió texto en español para traducir. Por favor, ingresa el texto completo.']);
    exit;
}

$adTextEs = $data['ad_text_es'];

// --- PROMPT MAESTRO para la IA ---
$prompt = "Actúa como un traductor experto en marketing digital y copywriter. Tu objetivo es traducir un texto publicitario del español al inglés, asegurándote de que la traducción no sea solo literal, sino estratégicamente optimizada para el marketing y las ventas en un público angloparlante.

Además de la traducción, proporciona breves notas estratégicas explicando las decisiones de traducción o los puntos clave que la hacen efectiva para el marketing.

--- TEXTO EN ESPAÑOL A TRADUCIR ---
\"\"\"
{$adTextEs}
\"\"\"

--- TU TAREA ESPECÍFICA ---
1.  Traduce el texto publicitario completo al inglés con un tono persuasivo y centrado en las ventas.
2.  Genera 2 o 3 'notas' estratégicas cortas (máximo 20 palabras cada una) que expliquen algún aspecto clave de la traducción o un consejo de marketing relacionado.

--- FORMATO DE RESPUESTA REQUERIDO ---
La respuesta DEBE ser un objeto JSON válido con la siguiente estructura y claves exactas:
{
    \"traduccion\": \"[Aquí va el texto publicitario traducido y optimizado al inglés]\",
    \"notas\": [
        \"[Nota estratégica 1, ej: 'Se adaptó el llamado a la acción para mayor urgencia.']\",
        \"[Nota estratégica 2, ej: 'Se usó lenguaje más directo para resonar con el público.']\"
    ]
}";

// URL de la API de Google Gemini
$apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-flash:generateContent?key=' . $apiKey;

$payload = [
    'contents' => [['parts' => [['text' => $prompt]]]],
    'generationConfig' => [
        'response_mime_type' => 'application/json',
        'response_schema' => [
            'type' => 'OBJECT',
            'properties' => [
                'traduccion' => ['type' => 'STRING'],
                'notas' => [
                    'type' => 'ARRAY',
                    'items' => ['type' => 'STRING']
                ]
            ]
        ]
    ]
];

// Inicializar cURL
$ch = curl_init($apiUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); // Devuelve la respuesta como string en lugar de imprimirla
curl_setopt($ch, CURLOPT_POST, true);           // Método POST
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload)); // Datos a enviar en formato JSON
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']); // Cabecera Content-Type

// Ejecutar la solicitud cURL y obtener la respuesta
$response = curl_exec($ch);
$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE); // Obtener el código HTTP
$curl_error = curl_error($ch); // Obtener cualquier error de cURL
curl_close($ch); // Cerrar la sesión cURL

// Manejo de errores de la solicitud cURL o respuesta HTTP
if ($response === false || $http_code !== 200) {
    http_response_code(500);
    echo json_encode([
        'error' => 'No se pudo obtener una respuesta de la IA. Posiblemente un problema de conexión o API Key.',
        'http_code' => $http_code,
        'curl_error' => $curl_error,
        'raw_response' => $response // Incluir respuesta cruda para depuración
    ]);
    exit;
}

// Decodificar la respuesta de la IA
$result = json_decode($response, true);

// Extraer el texto generado y limpiar de bloques de código Markdown
$generated_text = $result['candidates'][0]['content']['parts'][0]['text'] ?? null;
if ($generated_text) {
    $generated_text = str_replace(['```json', '```'], '', $generated_text);
}

// Intentar decodificar el JSON final.
$final_data = json_decode(trim($generated_text), true);

// Validar el formato del JSON final y su estructura esperada
if ($final_data === null || !isset($final_data['traduccion']) || !isset($final_data['notas'])) {
    http_response_code(500);
    echo json_encode([
        'error' => 'La IA devolvió un formato de JSON inválido o la estructura de datos esperada no está completa/correcta. Intenta de nuevo con tus datos.',
        'raw_ai_response_after_trim' => trim($generated_text),
        'decoded_data_debug' => $final_data
    ]);
    exit;
}

// Si todo es correcto, devolver los datos generados
echo json_encode($final_data);
?>