<?php
// db/file_helpers.php

// Función para eliminar un archivo del servidor de forma segura
// $file_url: La URL relativa del archivo como se guarda en la DB (ej. 'files/uploads/1/profile_photos/mi_foto.jpg')
// $tenant_id: El ID del inquilino (para construir la ruta base y por seguridad)
function deleteFileFromServer($file_url, $tenant_id) {
    if (empty($file_url)) {
        return; // No hay archivo que borrar
    }

    // Construir la ruta absoluta en el servidor
    // __DIR__ es el directorio actual (db/)
    // ../files/uploads/ es la ruta relativa desde db/ a files/uploads/
    $base_upload_dir = realpath(__DIR__ . '/../files/uploads/'); 
    
    // Si la $file_url viene con 'files/uploads/', la quitamos para que el realpath funcione bien con el tenant_id
    $relative_path_from_uploads = str_replace('files/uploads/', '', $file_url);

    // Si el tenant_id está en la URL (ej. files/uploads/1/profile_photos/...), normalizar
    // Esto es para que la ruta construida sea segura y verificable
    $file_path_on_server = $base_upload_dir . '/' . $relative_path_from_uploads;
    
    // Normalizar slashes y resolver rutas como ../
    $file_path_on_server = realpath($file_path_on_server);

    // SEGURIDAD CRÍTICA: Asegurarse de que el archivo a borrar ESTÉ DENTRO del directorio de uploads de ese tenant
    // y no sea una ruta manipulada para borrar archivos del sistema.
    $tenant_specific_upload_base = realpath($base_upload_dir . '/' . $tenant_id . '/');

    if ($file_path_on_server === false || strpos($file_path_on_server, $tenant_specific_upload_base) !== 0) {
        error_log("ADVERTENCIA DE SEGURIDAD: Intento de eliminar archivo fuera del directorio permitido para tenant ID " . $tenant_id . ": " . $file_url);
        return; // No permitir eliminación fuera del directorio del inquilino
    }

    if (file_exists($file_path_on_server) && is_file($file_path_on_server)) {
        if (unlink($file_path_on_server)) {
            error_log("Archivo antiguo eliminado con éxito para tenant " . $tenant_id . ": " . $file_path_on_server);
        } else {
            error_log("ERROR: No se pudo eliminar el archivo antiguo para tenant " . $tenant_id . ": " . $file_path_on_server);
        }
    } else {
        // error_log("ADVERTENCIA: Archivo no encontrado para eliminar: " . $file_path_on_server);
    }
}
?>