<?php
// db/invoice-update.php
ob_start();
error_reporting(0);
ini_set('display_errors', 0);

include_once 'db_connection.php';

$response = [];

try {
    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Error: JSON inválido recibido. ' . json_last_error_msg());
    }

    // Validar datos requeridos para la actualización de la factura
    if (empty($data['id']) || empty($data['invoice_date']) || empty($data['status']) || empty($data['client_id']) || !isset($data['items'])) {
        throw new Exception('Datos incompletos para la actualización de la factura. ID, fecha, estado, ID de cliente e ítems son requeridos.');
    }
    if (!is_array($data['items'])) { // Los ítems pueden estar vacíos si se eliminan todos
        throw new Exception('Los ítems de la factura deben ser un array.');
    }

    $invoice_db_id = filter_var($data['id'], FILTER_VALIDATE_INT);
    $client_id = filter_var($data['client_id'], FILTER_VALIDATE_INT);
    // lead_id se elimina por completo, ya no es relevante para facturas
    // No es necesario declarar $lead_id = null; aquí, ya que no se usará en la actualización.

    if ($invoice_db_id === false) {
        throw new Exception('ID de factura inválido.');
    }
    if ($client_id === false) {
        throw new Exception('ID de cliente inválido.');
    }

    $invoice_date = $data['invoice_date'];
    $due_date = !empty($data['due_date']) ? $data['due_date'] : null; // Campo opcional
    $status = trim($data['status']);
    $tax_rate = filter_var($data['tax_rate'] ?? 0.00, FILTER_VALIDATE_FLOAT);
    $items = $data['items']; // Array de [description, unit_price, quantity]

    // Validar que el estado sea uno de los permitidos para facturas
    $allowed_statuses = ['generada', 'enviada', 'cancelada', 'atrasada', 'pagada'];
    if (!in_array($status, $allowed_statuses)) {
        throw new Exception('Estado de factura inválido. Estados permitidos: ' . implode(', ', $allowed_statuses));
    }

    // Calcular subtotal y total_amount antes de actualizar
    $subtotal = 0;
    foreach ($items as $item) {
        if (!isset($item[0], $item[1], $item[2])) { // description, unit_price, quantity
            throw new Exception('Formato de ítem inválido. Cada ítem debe tener descripción, precio y cantidad.');
        }
        $item_price = filter_var($item[1], FILTER_VALIDATE_FLOAT);
        $item_quantity = filter_var($item[2], FILTER_VALIDATE_INT);
        if ($item_price === false || $item_quantity === false || $item_quantity < 0) { // Permitir cantidad 0 si se quiere representar un ítem sin costo
            throw new Exception('Precio o cantidad de ítem inválido.');
        }
        $subtotal += ($item_price * $item_quantity);
    }
    $total_amount = $subtotal + ($subtotal * ($tax_rate / 100));

    $pdo->beginTransaction();

    // Actualizar la factura principal (lead_id eliminado de la actualización)
    $stmt = $pdo->prepare(
        "UPDATE invoices
         SET client_id = ?, invoice_date = ?, due_date = ?, status = ?, tax_rate = ?, subtotal = ?, total_amount = ?, updated_at = NOW()
         WHERE id = ?"
    );
    $stmt->execute([
        $client_id,
        $invoice_date,
        $due_date,
        $status,
        $tax_rate,
        $subtotal,
        $total_amount,
        $invoice_db_id
    ]);

    // Eliminar ítems existentes de la factura para reinsertar los nuevos
    $stmt_delete_items = $pdo->prepare("DELETE FROM invoice_items WHERE invoice_id = ?");
    $stmt_delete_items->execute([$invoice_db_id]);

    // Insertar los nuevos ítems de la factura
    if (!empty($items)) {
        $stmt_insert_items = $pdo->prepare(
            "INSERT INTO invoice_items (invoice_id, description, unit_price, quantity)
             VALUES (?, ?, ?, ?)"
        );
        foreach ($items as $item) {
            $stmt_insert_items->execute([
                $invoice_db_id,
                $item[0], // description
                filter_var($item[1], FILTER_VALIDATE_FLOAT), // unit_price
                filter_var($item[2], FILTER_VALIDATE_INT) // quantity
            ]);
        }
    }

    $pdo->commit();

    $response = [
        'success' => true,
        'message' => 'Factura actualizada con éxito.',
        'id' => $invoice_db_id // Devuelve el ID de la DB
    ];

} catch (PDOException $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("PDO Error en invoice-update.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => 'Error de Base de Datos: ' . $e->getMessage()];
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    error_log("General Error en invoice-update.php: " . $e->getMessage());
    $response = ['success' => false, 'message' => $e->getMessage()];
}

ob_end_clean();
header('Content-Type: application/json');
echo json_encode($response);
exit;
?>
