<?php
// db/performance-read.php

// Inicia el búfer de salida. Es crucial para asegurar que las cabeceras se envíen correctamente.
ob_start();

// Habilitar errores para depuración. ¡IMPORTANTE: DESACTIVA ESTO EN PRODUCCIÓN!
// Mostrar todos los errores para facilitar la depuración.
error_reporting(E_ALL);
// Mostrar los errores directamente en el navegador. ¡DESACTIVA EN PRODUCCIÓN!
ini_set('display_errors', 1);

// Incluye el archivo de conexión a la base de datos.
// Asegúrate de que esta ruta es correcta y el archivo establece la conexión $pdo.
include_once 'db_connection.php'; 

// Prepara la estructura de respuesta inicial.
$response_data = [
    'success' => false,
    'message' => 'Error desconocido al cargar datos.',
    'data' => []
];

try {
    // Aquí puedes definir un tenant_id si tu sistema es multi-inquilino.
    // Si no usas un tenant_id, puedes eliminar esta variable y las cláusulas WHERE asociadas.
    $tenant_id = 1; // Ejemplo: filtra por el tenant_id 1.

    // Inicializa el array que contendrá todos los datos mensuales.
    $monthly_data = [];

    // --- Función Helper para obtener datos agrupados por mes ---
    // Esta función simplifica las consultas para contar o sumar datos por mes.
    // Se ha mejorado para aceptar una cláusula WHERE adicional y parámetros para sentencias preparadas.
    function fetchDataByMonth($pdo, $table, $date_column, $sum_column = null, $where_clause_suffix = '', $params = []) {
        $select_cols = "DATE_FORMAT({$date_column}, '%Y-%m') AS month_key, COUNT(id) AS count";
        if ($sum_column) {
            $select_cols .= ", SUM({$sum_column}) AS total_sum";
        }

        $sql = "SELECT {$select_cols} FROM {$table} WHERE 1=1 {$where_clause_suffix} GROUP BY month_key ORDER BY month_key ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // --- 1. Leads (Prospectos) ---
    // Cuenta todos los prospectos creados.
    $leads_raw = fetchDataByMonth($pdo, 'leads', 'created_at'); 
    foreach ($leads_raw as $row) {
        $month_key = $row['month_key'];
        // Inicializa el mes si no existe para evitar errores.
        if (!isset($monthly_data[$month_key])) $monthly_data[$month_key] = [];
        $monthly_data[$month_key]['leads'] = (int)$row['count'];
    }

    // --- 2. Clients (Clientes) ---
    // Cuenta todos los clientes creados.
    $clients_raw = fetchDataByMonth($pdo, 'clients', 'created_at'); 
    foreach ($clients_raw as $row) {
        $month_key = $row['month_key'];
        if (!isset($monthly_data[$month_key])) $monthly_data[$month_key] = [];
        $monthly_data[$month_key]['clientes'] = (int)$row['count'];
    }

    // --- 3. Services Sold (Servicios Vendidos) ---
    // Cuenta los ítems de factura SOLO si la factura está en estado 'pagada'.
    $services_sold_raw = $pdo->query("
        SELECT 
            DATE_FORMAT(i.invoice_date, '%Y-%m') AS month_key, 
            COUNT(ii.id) AS services_count
        FROM invoices i
        JOIN invoice_items ii ON i.id = ii.invoice_id
        WHERE LOWER(i.status) = 'pagada' -- ¡Usar LOWER para ignorar mayúsculas/minúsculas!
        GROUP BY month_key 
        ORDER BY month_key ASC
    ")->fetchAll(PDO::FETCH_ASSOC);
    foreach ($services_sold_raw as $row) {
        $month_key = $row['month_key'];
        if (!isset($monthly_data[$month_key])) $monthly_data[$month_key] = [];
        $monthly_data[$month_key]['serviciosVendidos'] = (int)$row['services_count'];
    }

    // --- 4. Facturación Mensual (Invoices) ---
    // Suma el total_amount SOLO de las facturas con estado 'pagada'.
    $facturacion_raw = fetchDataByMonth($pdo, 'invoices', 'invoice_date', 'total_amount', " AND LOWER(status) = 'pagada'"); // ¡Usar LOWER!
    foreach ($facturacion_raw as $row) {
        $month_key = $row['month_key'];
        if (!isset($monthly_data[$month_key])) $monthly_data[$month_key] = [];
        $monthly_data[$month_key]['facturacion'] = (float)$row['total_sum'];
        // La métrica 'facturasPagadas' se derivará de aquí.
        $monthly_data[$month_key]['facturasPagadas'] = (int)$row['count']; 
    }

    // --- 5. Cotizaciones Generadas (Estimates) ---
    // Cuenta TODAS las cotizaciones EXCEPTO las que tienen estado 'generada' (ignorando mayúsculas/minúsculas).
    $cotizaciones_raw = fetchDataByMonth($pdo, 'estimates', 'estimate_date', null, " AND LOWER(status) != 'generado'"); // ¡Usar LOWER!
    foreach ($cotizaciones_raw as $row) {
        $month_key = $row['month_key'];
        if (!isset($monthly_data[$month_key])) $monthly_data[$month_key] = [];
        $monthly_data[$month_key]['cotizacionesGeneradas'] = (int)$row['count'];
        // Si necesitas la suma de los montos de cotizaciones:
        // $monthly_data[$month_key]['cotizacionTotal'] = (float)$row['total_sum'];
    }

    // --- 6. Citas Nuevas (Unique Appointments) ---
    // Cuenta todas las citas nuevas.
    $citas_nuevas_raw = fetchDataByMonth($pdo, 'unique_appointments', 'appointment_date'); 
    foreach ($citas_nuevas_raw as $row) {
        $month_key = $row['month_key'];
        if (!isset($monthly_data[$month_key])) $monthly_data[$month_key] = [];
        $monthly_data[$month_key]['citasNuevas'] = (int)$row['count'];
    }

    // --- 7. Citas Recurrentes (Recurring Appointments) ---
    // Cuenta todas las citas recurrentes.
    $citas_recurrentes_raw = fetchDataByMonth($pdo, 'recurring_appointments', 'next_appointment_date'); 
    foreach ($citas_recurrentes_raw as $row) {
        $month_key = $row['month_key'];
        if (!isset($monthly_data[$month_key])) $monthly_data[$month_key] = [];
        $monthly_data[$month_key]['citasRecurrentes'] = (int)$row['count'];
    }

    // --- Asegurar Consistencia de Datos para todos los Meses ---
    // Este bloque garantiza que todos los meses en el rango tengan todas las métricas,
    // rellenando con 0 si no hay datos específicos para alguna.
    $all_month_keys = array_keys($monthly_data);
    $min_month = $all_month_keys ? min($all_month_keys) : date('Y-m');
    $max_month = $all_month_keys ? max($all_month_keys) : date('Y-m');

    $start_year = (int)substr($min_month, 0, 4);
    $start_month = (int)substr($min_month, 5, 2);
    $end_year = (int)substr($max_month, 0, 4);
    $end_month = (int)substr($max_month, 5, 2);

    $current_date = new DateTime("{$start_year}-{$start_month}-01");
    $end_date = new DateTime("{$end_year}-{$end_month}-01");

    while ($current_date <= $end_date) {
        $month_key = $current_date->format('Y-m');
        if (!isset($monthly_data[$month_key])) {
            $monthly_data[$month_key] = []; 
        }
        
        // Asigna 0 a cualquier métrica que no tenga datos para este mes.
        $monthly_data[$month_key]['leads'] = $monthly_data[$month_key]['leads'] ?? 0; 
        $monthly_data[$month_key]['clientes'] = $monthly_data[$month_key]['clientes'] ?? 0; 
        $monthly_data[$month_key]['serviciosVendidos'] = $monthly_data[$month_key]['serviciosVendidos'] ?? 0; 
        $monthly_data[$month_key]['facturacion'] = (float)($monthly_data[$month_key]['facturacion'] ?? 0.0); 
        $monthly_data[$month_key]['cotizacionesGeneradas'] = $monthly_data[$month_key]['cotizacionesGeneradas'] ?? 0; 
        $monthly_data[$month_key]['facturasPagadas'] = $monthly_data[$month_key]['facturasPagadas'] ?? 0; 
        $monthly_data[$month_key]['citasNuevas'] = $monthly_data[$month_key]['citasNuevas'] ?? 0; 
        $monthly_data[$month_key]['citasRecurrentes'] = $monthly_data[$month_key]['citasRecurrentes'] ?? 0; 

        $current_date->modify('+1 month');
    }

    // Ordena los datos por la clave del mes (año-mes).
    ksort($monthly_data);

    // Si todo salió bien, marca la respuesta como exitosa y adjunta los datos.
    $response_data = ['success' => true, 'data' => $monthly_data];

} catch (PDOException $e) {
    // Registra el error de la base de datos para depuración.
    error_log("Error PDO al cargar rendimiento: " . $e->getMessage());
    // Mensaje para el usuario final (más genérico en producción).
    $response_data['message'] = 'Error de Base de Datos al cargar rendimiento: ' . $e->getMessage();
} catch (Exception $e) {
    // Registra cualquier otro error inesperado.
    error_log("Error general al cargar rendimiento: " . $e->getMessage());
    // Mensaje para el usuario final.
    $response_data['message'] = 'Error inesperado al cargar rendimiento: ' . $e->getMessage();
}

// Limpia el búfer de salida y envía las cabeceras HTTP.
ob_end_clean();
header('Content-Type: application/json');
// Codifica el array de respuesta a JSON y lo imprime.
echo json_encode($response_data);
// Termina la ejecución del script.
exit;