<?php include 'files/guardiankey.php'; ?>
<?php
ob_start(); // Iniciar el búfer de salida al inicio del script
session_start();

// ***** IMPORTANTE: ESTAS LÍNEAS ESTÁN AQUÍ PARA DEPURAR. *****
// UNA VEZ QUE FUNCIONE, CÁMBIALAS A: error_reporting(0); ini_set('display_errors', 0);
error_reporting(E_ALL); // Reporta todos los errores
ini_set('display_errors', 1); // Muestra los errores en la pantalla
// *************************************************************

// Incluimos el archivo de conexión a la base de datos
require_once 'db/db_connection.php'; 

// --- Lógica PHP para obtener datos de servicios directamente ---
$servicios_data_from_db = [];
$monthly_sold_services_count = 0; // Este valor debe venir de una consulta a la DB que aún no tienes

try {
    $stmt = $pdo->query("SELECT id, name, price, category, description, show_on_web FROM services ORDER BY name ASC");
    $db_services_rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($db_services_rows as $row) {
        // Asegúrate que el ID en PHP y JS sea consistente. Usaremos 'service-<ID_DB>'
        $servicios_data_from_db['service-' . $row['id']] = [
            'id' => 'service-' . $row['id'], // ID para uso en DOM y JS
            'db_id' => $row['id'], // ID original de la base de datos
            'nombre' => $row['name'],
            'precio' => $row['price'],
            'categoria' => $row['category'],
            'descripcion' => $row['description'],
            'show_on_web' => (bool)(int)$row['show_on_web']
        ];
    }
} catch (PDOException $e) {
    error_log("Error PDO al cargar servicios en services.php: " . $e->getMessage());
} catch (Exception $e) {
    error_log("Error general al cargar servicios en services.php: " . $e->getMessage());
}

// NO cargamos las categorías desde la tabla 'services' aquí en PHP
// sino que haremos una llamada AJAX en JS para obtenerlas de la nueva API.
$servicios_json = json_encode($servicios_data_from_db);
$monthly_sold_services_json = json_encode($monthly_sold_services_count); // Este sigue siendo 0 por ahora

ob_end_clean();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>LocalCRM ToolKit Dashboard | Servicios | OrozDesign Multiemdia</title>
    <meta name="description" content="Administra tus servicios en LocalCRM: crea, edita y organiza tu catálogo, define precios y presenta opciones claras a tus clientes">
    <meta name="robots" content="noindex, nofollow">

    <link rel="icon" type="image/png" href="img/favicon.png">
    <link rel="apple-touch-icon" href="img/apple-touch-icon.png">

    <?php include 'files/gtm-head.php'; ?>

    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Barlow:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <script src="https://unpkg.com/lucide@latest"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <link rel="stylesheet" href="style.css">
    <script src="files/header-manager.js"></script>

</head>
<body data-page-title="CATÁLOGO DE SERVICIOS"
      data-page-subtitle="GESTIÓN COMPLETA DE TU OFERTA COMERCIAL"
      data-page-icon="briefcase">

    <div id="toast-container" class="toast-container"></div>
        
<?php include 'files/gtm-body.php'; ?>

<div class="relative min-h-screen md:flex">

    <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>

    <?php include 'menu.php'; ?>

    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
            <button id="mobile-menu-button" class="md:hidden text-gray-600 hover:text-gray-800">
                <i data-lucide="menu" class="w-6 h-6"></i>
            </button>
            <div class="page-header-container">
                <h2 id="page-title"></h2>
                <p id="page-subtitle"></p>
            </div>
        </header>

        <div id="content-area" class="p-4 md:p-8 space-y-8">
            <section id="servicios" class="dashboard-section">
                <div class="bg-white p-6 rounded-xl shadow-md">
                    <div class="flex flex-col md:flex-row justify-between items-start md:items-center mb-6 gap-4">
                        <div>
                            <h3 class="text-2xl font-extrabold text-gray-800 flex items-center gap-2">
                                <i data-lucide="package" class="w-7 h-7 text-[var(--color-primary)]"></i> INVENTARIO DE SERVICIOS
                            </h3>
                            <p class="text-gray-500 text-sm mt-1 uppercase">VISUALIZA Y EDITA TUS PRODUCTOS Y SERVICIOS.</p>
                        </div>
                        <div class="flex flex-col sm:flex-row items-center gap-4 w-full md:w-auto">
                            <button class="btn-secondary font-bold py-2 px-4 rounded-lg flex items-center w-full sm:w-auto justify-center uppercase" onclick="openPanel('addServicioPanel')">
                                <i data-lucide="plus" class="w-5 h-5 mr-2"></i> AGREGAR SERVICIO
                            </button>
                            <button class="btn-primary font-bold py-2 px-4 rounded-lg flex items-center w-full sm:w-auto justify-center uppercase text-sm" onclick="openPanel('manageCategoriesPanel')">
                                <i data-lucide="tag" class="w-4 h-4 mr-2"></i> ADMINISTRAR CATEGORÍAS
                            </button>
                            <button id="download-servicios-btn" class="btn-primary font-bold py-2 px-4 rounded-lg flex items-center w-full sm:w-auto justify-center uppercase text-sm">
                                <i data-lucide="download" class="w-4 h-4 mr-2"></i> DESCARGAR
                            </button>
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
    <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-primary)]">
        <i data-lucide="package" class="w-12 h-12 text-gray-700"></i>
        <div>
            <h3 class="text-lg font-extrabold text-gray-500 mb-1">TOTAL DE SERVICIOS</h3>
            <p id="total-services-count" class="text-5xl font-bold text-[var(--color-secondary)]">0</p>
            <p class="text-sm text-gray-400 mt-1">EN EL CATÁLOGO</p>
        </div>
    </div>
    <div class="stat-card bg-white p-6 rounded-xl shadow-md flex items-center space-x-4 border-l-4 border-[var(--color-highlight)]">
        <i data-lucide="wallet" class="w-12 h-12 text-gray-700"></i>
        <div>
            <h3 class="text-lg font-extrabold text-gray-500 mb-1">SERVICIOS VENDIDOS ESTE MES</h3>
            <p id="monthly-sold-services-count" class="text-5xl font-bold text-[var(--color-secondary)]">0</p>
            <p class="text-sm text-gray-400 mt-1 uppercase">Basado en registros de clientes</p>
        </div>
    </div>
</div>

                    <div class="flex flex-col md:flex-row gap-4 mb-6">
                        <div class="relative flex-grow">
                            <input type="text" id="service-search" placeholder="Buscar Por Nombre o Descripción..." class="w-full p-3 pl-10 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]">
                            <i data-lucide="search" class="absolute left-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"></i>
                        </div>
                        <select id="service-filter-category" class="w-full md:w-48 p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white">
                            <option value="all">TODAS LAS CATEGORÍAS</option>
                            </select>
                        <div class="view-toggle inline-flex bg-gray-200 rounded-lg p-1">
                            <button id="grid-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 active uppercase" title="Vista de Cuadrícula">
                                <i data-lucide="layout-grid" class="w-5 h-5"></i>
                            </button>
                            <button id="list-view-btn" class="px-3 py-1 rounded-md text-sm font-semibold transition-colors duration-200 uppercase" title="Vista de Lista">
                                <i data-lucide="list" class="w-5 h-5"></i>
                            </button>
                        </div>
                    </div>

                    <div id="service-grid-view" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    </div>

                    <div id="service-list-view" class="overflow-x-auto hidden">
                        <table class="min-w-full bg-white responsive-invoices-table">
                            <thead class="bg-gray-50">
                                <tr class="text-left text-gray-500 uppercase text-sm">
                                    <th class="py-3 px-6 font-semibold">NOMBRE</th>
                                    <th class="py-3 px-6 font-semibold hidden md:table-cell">CATEGORÍA</th>
                                    <th class="py-3 px-6 font-semibold">PRECIO</th>
                                    <th class="py-3 px-6 font-semibold">VISIBILIDAD WEB</th>
                                    <th class="py-3 px-6 font-semibold text-center actions-cell">ACCIONES</th>
                                </tr>
                            </thead>
                            <tbody class="text-gray-700 text-sm" id="servicesTableBody">
                            </tbody>
                        </table>
                    </div>

                </div>
            </section>
        </div>
    </main>
</div>

<div id="addServicioPanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">AGREGAR NUEVO SERVICIO</h3>
            <button onclick="closePanel('addServicioPanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <form id="add-servicio-form" class="flex-grow overflow-y-auto pr-2 -mr-2">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-4">
                <div class="md:col-span-2">
                    <label for="servicio-nombre" class="block text-gray-700 text-sm font-bold mb-2 uppercase">NOMBRE DEL SERVICIO</label>
                    <input type="text" id="servicio-nombre" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. Limpieza Semanal" required>
                </div>
                <div>
                    <label for="servicio-precio" class="block text-gray-700 text-sm font-bold mb-2 uppercase">PRECIO</label>
                    <input type="number" step="0.01" id="servicio-precio" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. 50.00" required>
                </div>
                <div>
                    <label for="servicio-categoria" class="block text-gray-700 text-sm font-bold mb-2 uppercase">CATEGORÍA</label>
                    <select id="servicio-categoria" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white">
                        <option value="">Selecciona Una Categoría</option>
                        </select>
                </div>
                <div class="md:col-span-2">
                    <label for="servicio-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label>
                    <textarea id="servicio-descripcion" rows="3" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Descripción Del Servicio"></textarea>
                </div>
                <div class="md:col-span-2 flex items-center pt-2">
                    <input type="checkbox" id="servicio-mostrar-web" class="h-5 w-5 text-[var(--color-primary)] focus:ring-[var(--color-highlight)] border-gray-300 rounded mr-2">
                    <label for="servicio-mostrar-web" class="text-gray-700 text-sm font-bold uppercase">MOSTRAR EN MI PÁGINA WEB</label>
                </div>
            </div>
            <div class="h-4"></div></form>
        <div class="flex justify-end space-x-4 pt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('addServicioPanel')">CANCELAR</button>
            <button type="submit" form="add-servicio-form" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase">GUARDAR SERVICIO</button>
        </div>
    </div>
</div>

<div id="confirmDeleteModal" class="fixed inset-0 bg-gray-900 bg-opacity-75 flex items-center justify-center hidden z-50">
    <div class="bg-white p-8 rounded-xl shadow-2xl w-full max-w-sm m-4 transform transition-all duration-300 scale-95 opacity-0 text-center">
        <div class="flex justify-center mb-4">
            <i data-lucide="alert-triangle" class="w-16 h-16 text-red-500"></i>
        </div>
        <h3 class="text-2xl font-bold text-[var(--color-primary)] mb-4 uppercase">CONFIRMAR ELIMINACIÓN</h3>
        <p class="text-gray-700 mb-6 uppercase">¿ESTÁS SEGURO DE QUE DESEAS ELIMINAR ESTE <span id="confirm-item-type" class="font-semibold">ELEMENTO</span>? ESTA ACCIÓN NO SE PUEDE DESHACER.</p>
        <div class="flex justify-center space-x-4">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closeModal('confirmDeleteModal')">CANCELAR</button>
            <button type="button" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase" id="confirm-delete-button">CONFIRMAR</button>
        </div>
    </div>
</div>

<div id="viewServicioPanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">DETALLE DEL SERVICIO</h3>
            <button onclick="closePanel('viewServicioPanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <form id="view-servicio-form" class="flex-grow overflow-y-auto pr-2 -mr-2">
            <input type="hidden" id="view-servicio-id">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-x-6 gap-y-4">
                <div class="md:col-span-2">
                    <label for="view-servicio-nombre" class="block text-gray-700 text-sm font-bold mb-2 uppercase">NOMBRE DEL SERVICIO</label>
                    <input type="text" id="view-servicio-nombre" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Nombre Del servicio">
                </div>
                <div>
                    <label for="view-servicio-precio" class="block text-gray-700 text-sm font-bold mb-2 uppercase">PRECIO</label>
                    <input type="number" step="0.01" id="view-servicio-precio" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. 50.00">
                </div>
                <div>
                    <label for="view-servicio-categoria" class="block text-gray-700 text-sm font-bold mb-2 uppercase">CATEGORÍA</label>
                    <select id="view-servicio-categoria" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)] bg-white">
                        <option value="">Selecciona Una Categoría</option>
                        </select>
                </div>
                <div class="md:col-span-2">
                    <label for="view-servicio-descripcion" class="block text-gray-700 text-sm font-bold mb-2 uppercase">DESCRIPCIÓN</label>
                    <textarea id="view-servicio-descripcion" rows="3" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Descripción Del Servicio"></textarea>
                </div>
                <div class="md:col-span-2 flex items-center pt-2">
                    <input type="checkbox" id="view-servicio-mostrar-web" class="h-5 w-5 text-[var(--color-primary)] focus:ring-[var(--color-highlight)] border-gray-300 rounded mr-2">
                    <label for="view-servicio-mostrar-web" class="text-gray-700 text-sm font-bold uppercase">MOSTRAR EN MI PÁGINA WEB</label>
                </div>
            </div>
            <div class="h-4"></div></form>
        <div class="flex justify-end space-x-4 pt-4 border-t border-gray-200 mt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('viewServicioPanel')">CANCELAR</button>
            <button type="submit" form="view-servicio-form" class="btn-primary font-bold py-2 px-4 rounded-lg uppercase">GUARDAR CAMBIOS</button>
        </div>
    </div>
</div>

<div id="manageCategoriesPanel" class="fixed inset-y-0 right-0 w-full md:w-1/2 bg-white shadow-lg transform translate-x-full transition-transform duration-300 ease-in-out z-50 overflow-y-auto">
    <div class="p-8 flex flex-col h-full">
        <div class="flex justify-between items-center mb-6 flex-shrink-0">
            <h3 class="text-2xl font-bold text-[var(--color-primary)]">ADMINISTRAR CATEGORÍAS</h3>
            <button onclick="closePanel('manageCategoriesPanel')" class="text-gray-400 hover:text-gray-600"><i data-lucide="x" class="w-6 h-6"></i></button>
        </div>
        <div class="flex-grow overflow-y-auto pr-2 -mr-2">
            <h4 class="text-xl font-bold text-gray-700 mb-4 uppercase">Agregar Nueva Categoría</h4>
            <form id="add-category-form" class="mb-8">
                <div class="mb-4">
                    <label for="new-category-name" class="block text-gray-700 text-sm font-bold mb-2 uppercase">Nombre De La Categoría</label>
                    <input type="text" id="new-category-name" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-[var(--color-highlight)]" placeholder="Ej. Contabilidad" required>
                </div>
                <button type="submit" class="btn-secondary font-bold py-2 px-4 rounded-lg uppercase">AÑADIR CATEGORÍA</button>
            </form>

            <h4 class="text-xl font-bold text-gray-700 mb-4 uppercase">Categorías Existentes</h4>
            <div id="category-list-container" class="space-y-2">
                <p class="text-gray-500 text-sm italic">Cargando Categorías...</p>
            </div>
        </div>
        <div class="flex justify-end space-x-4 pt-6 flex-shrink-0">
            <button type="button" class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-bold py-2 px-4 rounded-lg uppercase" onclick="closePanel('manageCategoriesPanel')">CERRAR</button>
        </div>
    </div>
</div>


<script>
// Variable global para almacenar los servicios cargados desde PHP
let serviciosActuales = <?php echo $servicios_json; ?>;
// NUEVO: Variable para almacenar la lista de categorías (ahora se cargará por AJAX)
let allCategories = []; 
// NUEVO: Variable para almacenar el contador de servicios vendidos al mes
let monthlySoldServicesCount = 0; // Se inicializa en 0 y se actualiza más abajo si hay un valor PHP


document.addEventListener('DOMContentLoaded', function() {
    // --- CONSOLE.LOGS PARA DEPURACIÓN ---
    // showToast("--- Depuración de services.php ---", 'info');
    // console.log("serviciosActuales (datos de servicios de la DB al cargar PHP):", serviciosActuales); // Reemplazado
    // showToast("monthlySoldServicesCount (contador mensual desde PHP):" + monthlySoldServicesCount, 'info');
    // --- FIN CONSOLE.LOGS PARA DEPURACIÓN ---


    // --- VARIABLES DOM LOCALES ---
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const sidebar = document.getElementById('sidebar');
    const sidebarOverlay = document.getElementById('sidebar-overlay');
    const serviceSearch = document.getElementById('service-search');
    const serviceFilterCategory = document.getElementById('service-filter-category');
    const serviceGridView = document.getElementById('service-grid-view');
    const serviceListView = document.getElementById('service-list-view');
    const servicesTableBody = document.getElementById('servicesTableBody');
    const gridViewBtn = document.getElementById('grid-view-btn');
    const listViewBtn = document.getElementById('list-view-btn');
    const downloadServiciosBtn = document.getElementById('download-servicios-btn');
    const confirmDeleteButton = document.getElementById('confirm-delete-button');
    const confirmItemTypeSpan = document.getElementById('confirm-item-type');
    const addServicioForm = document.getElementById('add-servicio-form');
    const viewServicioForm = document.getElementById('view-servicio-form');
    const totalServicesCountElement = document.getElementById('total-services-count');
    const servicioMostrarWebCheckbox = document.getElementById('servicio-mostrar-web');
    const viewServicioMostrarWebCheckbox = document.getElementById('view-servicio-mostrar-web');
    // NUEVAS referencias DOM para categorías
    const addServiceCategorySelect = document.getElementById('servicio-categoria');
    const viewServiceCategorySelect = document.getElementById('view-servicio-categoria');
    const newCategoryNameInput = document.getElementById('new-category-name');
    const addCategoryForm = document.getElementById('add-category-form');
    const categoryListContainer = document.getElementById('category-list-container');
    // NUEVO: Referencia DOM para el contador de servicios vendidos al mes
    const monthlySoldServicesCountElement = document.getElementById('monthly-sold-services-count');


    let currentServiceView = 'grid';
    
    // Elemento de overlay para los paneles laterales (creado dinámicamente)
    const panelOverlay = document.createElement('div');
    panelOverlay.id = 'panel-overlay';
    panelOverlay.className = 'fixed inset-0 bg-gray-900 bg-opacity-75 z-40 hidden';
    document.body.appendChild(panelOverlay);


    // --- FUNCIONES HELPER GLOBALES ---

    // MODIFICADO: showCustomNotification ahora es un alias de showToast
    // La definición real de showToast está en header-manager.js
    // window.showCustomNotification = showToast; // Ya se hace en header-manager.js

    // Abre un panel lateral (MODIFICADO para incluir lógica de inicialización)
    window.openPanel = function(panelId) {
        const panel = document.getElementById(panelId);
        if (!panel) { showToast(`Error: Panel con ID "${panelId}" no encontrado.`, 'error'); return; }
        panel.classList.remove('translate-x-full');
        panelOverlay.classList.remove('hidden'); // Muestra el overlay
        // console.log(`[services.php] Panel ${panelId} abierto.`); // Reemplazado
        
        // Lógica específica para el panel de agregar servicio (si aplica, p.ej. resetear formulario)
        if (panelId === 'addServicioPanel') {
            addServicioForm.reset();
            populateCategoryDropdowns(); // Llenar el dropdown de categorías al abrir
        } else if (panelId === 'manageCategoriesPanel') {
            newCategoryNameInput.value = ''; // Limpiar input al abrir
            renderCategoryList(); // Renderizar la lista de categorías existentes
        }
    };

    // Cierra un panel lateral (MODIFICADO para incluir el nuevo panel)
    window.closePanel = function(panelId) {
        const panel = document.getElementById(panelId);
        if (!panel) { showToast(`Error: Panel con ID "${panelId}" no encontrado para cerrar.`, 'error'); return; }
        panel.classList.add('translate-x-full');
        panelOverlay.classList.add('hidden'); // Oculta el overlay
        // console.log(`[services.php] Panel ${panelId} cerrado.`); // Reemplazado
    };

    // Abre un modal (solo para confirmDeleteModal)
    window.openModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) { showToast(`Error: Modal con ID "${modalId}" no encontrado.`, 'error'); return; }
        const modalBox = modal.querySelector('div:first-of-type');
        modal.classList.remove('hidden');
        setTimeout(() => { modalBox.classList.remove('scale-95', 'opacity-0'); }, 50);
        // console.log(`[services.php] Modal ${modalId} abierto.`); // Reemplazado
    };

    // Cierra un modal (solo para confirmDeleteModal)
    window.closeModal = function(modalId) {
        const modal = document.getElementById(modalId);
        if (!modal) { showToast(`Error: Modal con ID "${modalId}" no encontrado para cerrar.`, 'error'); return; }
        const modalBox = modal.querySelector('div:first-of-type');
        modalBox.classList.add('scale-95', 'opacity-0');
        setTimeout(() => { modal.classList.add('hidden'); }, 300);
        // console.log(`[services.php] Modal ${modalId} cerrado.`); // Reemplazado
    };
    
    // Abre el modal de confirmación de eliminación (sin cambios)
    window.openConfirmDeleteModal = function(itemId, itemType) {
        if (!confirmDeleteButton || !confirmItemTypeSpan) { showToast("Error: Elementos del modal de confirmación de eliminación no encontrados.", 'error'); return; }
        confirmDeleteButton.dataset.itemId = itemId; confirmDeleteButton.dataset.itemType = itemType;
        confirmItemTypeSpan.textContent = itemType.toUpperCase();
        openModal('confirmDeleteModal');
    };

    // Descarga datos como CSV (sin cambios, ya era funcional)
    function descargarComoCSV(dataObject, filename) {
        const data = Object.values(dataObject);
        if (data.length === 0) { showToast('No hay datos para descargar.', 'info'); return; }
        showToast('Preparando descarga...', 'info'); // Toast de info
        const escapeCSV = (value) => {
            if (value === null || value === undefined) { return ''; }
            value = String(value);
            if (value.search(/("|,|\n)/g) >= 0) { return `"${value.replace(/"/g, '""')}"`; }
            return value;
        };
        const headers = ['db_id', 'nombre', 'precio', 'categoria', 'descripcion', 'show_on_web'];
        const csvRows = [headers.map(h => escapeCSV(h.replace(/_/g, ' ').toUpperCase())).join(',')];
        data.forEach(item => {
            const values = headers.map(header => escapeCSV(item[header]));
            csvRows.push(values.join(','));
        });
        const csvContent = csvRows.join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        if (link.download !== undefined) {
            const url = URL.createObjectURL(blob);
            link.setAttribute('href', url); link.setAttribute('download', filename);
            link.style.visibility = 'hidden'; document.body.appendChild(link);
            link.click(); document.body.removeChild(link);
            showToast('Descarga completada con éxito.', 'success'); // Toast de éxito
        }
    }

    // Abre el panel de Ver/Editar Servicio (MODIFICADO: openPanel en lugar de openModal y ahora usa dropdowns)
    window.openViewServicioModal = function(servicioId) {
        const servicio = serviciosActuales[servicioId];
        if (!servicio) { showToast('Servicio no encontrado para editar.', 'error'); return; }
        document.getElementById('view-servicio-id').value = servicio.db_id;
        document.getElementById('view-servicio-nombre').value = servicio.nombre;
        document.getElementById('view-servicio-precio').value = servicio.precio;
        // NUEVO: Llenar y seleccionar en el dropdown de edición
        populateCategoryDropdowns(servicio.categoria);
        document.getElementById('view-servicio-descripcion').value = servicio.descripcion || '';
        viewServicioMostrarWebCheckbox.checked = servicio.show_on_web;
        openPanel('viewServicioPanel'); // ¡Cambiado a openPanel!
    };

    // Setup de listeners (sin cambios en la lógica interna)
    function setupDeleteListeners() {
        document.querySelectorAll('.delete-btn').forEach(button => {
            button.removeEventListener('click', handleDeleteButtonClick);
            button.addEventListener('click', handleDeleteButtonClick);
        });
    }
    function handleDeleteButtonClick(e) {
        e.stopPropagation();
        const itemId = this.dataset.itemId; const itemType = this.dataset.itemType;
        openConfirmDeleteModal(itemId, itemType);
    }

    function setupEditListeners() {
        document.querySelectorAll('.edit-servicio-btn').forEach(button => {
            button.removeEventListener('click', handleEditServicioButtonClick);
            button.addEventListener('click', handleEditServicioButtonClick);
        });
    }
    function handleEditServicioButtonClick(e) {
        e.stopPropagation();
        const servicioId = this.dataset.servicioId;
        openViewServicioModal(servicioId);
    }
    
    function toggleSidebar() {
        if (sidebar) sidebar.classList.toggle('-translate-x-full');
        if (sidebarOverlay) sidebarOverlay.classList.toggle('hidden');
    }

    function updateTotalServicesDisplay() {
        if (totalServicesCountElement) {
            const total = Object.keys(serviciosActuales).length;
            totalServicesCountElement.textContent = total.toString();
        }
    }

    // NUEVO: Función para actualizar el contador de servicios vendidos al mes
    function updateMonthlySoldServicesDisplay() {
        if (monthlySoldServicesCountElement) {
            monthlySoldServicesCountElement.textContent = monthlySoldServicesCount.toString();
        }
    }

    // --- NUEVO: Función para cargar las categorías desde la API ---
    async function fetchCategories() {
        try {
            const response = await fetch('db/services-categories-read.php'); // Usa el nuevo nombre de archivo
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            const data = await response.json();
            if (data.success) {
                // allCategories ahora almacenará objetos {id, name}
                allCategories = data.categories.map(cat => ({ id: cat.id, name: cat.name }));
                // console.log("Categorías cargadas exitosamente:", allCategories); // Reemplazado
            } else {
                showToast('Error al cargar categorías: ' + (data.message || 'Error desconocido'), 'error');
                console.error('Error al cargar categorías:', data.message);
            }
        } catch (error) {
            showToast('Error de conexión al cargar las categorías.', 'error');
            console.error('Fetch categories error:', error);
        } finally {
            // Asegurarse de que los dropdowns y la lista se actualicen
            populateCategoryFilter();
            populateCategoryDropdowns();
            renderCategoryList(); // Renderizar lista de categorías al cargar el panel
        }
    }


    // Función para poblar el filtro principal de categorías
    function populateCategoryFilter() {
        // Asegúrate de que allCategories tenga un formato consistente (array de objetos {id, name})
        const filterCategories = new Set(allCategories.map(cat => cat.name)); // Usar el 'name' para el filtro
        const sortedCategories = Array.from(filterCategories).sort();
        serviceFilterCategory.innerHTML = '<option value="all">TODAS LAS CATEGORÍAS</option>';
        sortedCategories.forEach(categoryName => {
            const option = document.createElement('option');
            option.value = categoryName;
            option.textContent = categoryName.toUpperCase();
            serviceFilterCategory.appendChild(option);
        });
    }

    // NUEVO: Función para poblar los dropdowns de categoría en los formularios de añadir/editar
    function populateCategoryDropdowns(selectedCategoryName = '') {
        addServiceCategorySelect.innerHTML = '<option value="">Selecciona Una Categoría</option>';
        viewServiceCategorySelect.innerHTML = '<option value="">Selecciona Una Categoría</option>';

        const sortedCategories = [...allCategories].sort((a, b) => a.name.localeCompare(b.name));

        sortedCategories.forEach(category => {
            const optionAdd = document.createElement('option');
            optionAdd.value = category.name; // Usar el nombre de la categoría como valor
            optionAdd.textContent = category.name;
            if (category.name === selectedCategoryName) {
                optionAdd.selected = true;
            }
            addServiceCategorySelect.appendChild(optionAdd);

            const optionView = document.createElement('option');
            optionView.value = category.name; // Usar el nombre de la categoría como valor
            optionView.textContent = category.name;
            if (category.name === selectedCategoryName) {
                optionView.selected = true;
            }
            viewServiceCategorySelect.appendChild(optionView);
        });
    }

    // NUEVO: Función para renderizar la lista de categorías en el panel de administración
    function renderCategoryList() {
        categoryListContainer.innerHTML = ''; // Limpiar lista existente
        if (allCategories.length === 0) {
            categoryListContainer.innerHTML = '<p class="text-gray-500 text-sm italic">No Hay Categorías Definidas. Añade Una.</p>';
            return;
        }

        const sortedCategories = [...allCategories].sort((a, b) => a.name.localeCompare(b.name));
        sortedCategories.forEach(category => {
            const categoryItem = document.createElement('div');
            // Usamos category.id para el data-id del botón de eliminar
            categoryItem.className = 'flex items-center justify-between p-3 bg-gray-100 rounded-lg border border-gray-200';
            categoryItem.innerHTML = `
                <span class="text-gray-800 font-semibold uppercase">${category.name}</span>
                <button type="button" class="text-red-500 hover:text-red-700 delete-category-btn" data-category-id="${category.id}" title="Eliminar categoría">
                    <i data-lucide="trash-2" class="w-5 h-5"></i>
                </button>
            `;
            categoryListContainer.appendChild(categoryItem);
        });
        lucide.createIcons({ container: categoryListContainer }); // Renderizar iconos Lucide

        // Adjuntar listeners para botones de eliminar categoría (AHORA REAL)
        categoryListContainer.querySelectorAll('.delete-category-btn').forEach(button => {
            button.addEventListener('click', async function() {
                const categoryIdToDelete = this.dataset.categoryId;
                showToast(`Eliminando categoría...`, 'info');
                try {
                    const response = await fetch('db/services-categories-delete.php', { // Usa el nuevo nombre de archivo
                        method: 'POST', // O DELETE si tu API lo soporta y lo configuraste así
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify({ id: categoryIdToDelete })
                    });
                    const data = await response.json();
                    if (data.success) {
                        showToast('Categoría eliminada con éxito.', 'success');
                        await fetchCategories(); // Recargar categorías del servidor
                    } else {
                        showToast('Error al eliminar categoría: ' + (data.message || 'Error desconocido.'), 'error');
                    }
                } catch (error) {
                    showToast('Error de conexión al eliminar la categoría.', 'error');
                    console.error('Delete category error:', error);
                }
            });
        });
    }


    function renderServiceCard(service) {
        const card = document.createElement('div');
        card.className = 'bg-white rounded-xl shadow-md overflow-hidden flex flex-col border border-gray-200';
        card.dataset.servicioId = service.id;
        card.dataset.itemId = service.id;
        const visibilityIcon = service.show_on_web ? '<i data-lucide="eye" class="w-5 h-5 text-green-500" title="Visible en Web"></i>' : '<i data-lucide="eye-off" class="w-5 h-5 text-gray-500" title="Oculto en Web"></i>';
        card.innerHTML = `
            <div class="bg-[var(--color-primary)] text-white p-4 flex items-center justify-between">
                <div class="flex items-center gap-2 flex-grow min-w-0">
                    <i data-lucide="clipboard-check" class="w-5 h-5 flex-shrink-0"></i>
                    <h3 class="font-bold text-lg uppercase truncate">${service.nombre}</h3>
                </div>
                <div class="flex-shrink-0 ml-2">${visibilityIcon}</div>
            </div>
            <div class="p-6 flex-grow flex flex-col">
                <p class="text-gray-600 uppercase text-sm mb-2 font-semibold">PRECIO DEL SERVICIO:</p>
                <p class="text-[var(--color-secondary)] font-extrabold text-3xl mb-4">$${parseFloat(service.precio).toFixed(2)}</p>
                <p class="text-gray-700 text-sm flex-grow mb-4 line-clamp-3">${service.descripcion || 'Sin descripción detallada.'}</p>
                <p class="text-gray-500 text-xs uppercase mt-auto">Categoría: <span class="font-semibold">${service.categoria || 'N/A'}</span></p>
            </div>
            <div class="bg-gray-100 p-3 flex justify-end space-x-3 border-t border-gray-200">
                <button class="bg-gray-200 hover:bg-gray-300 text-gray-800 font-semibold py-2 px-4 rounded-lg text-sm uppercase edit-servicio-btn" data-servicio-id="${service.id}">EDITAR</button>
                <button class="p-2 bg-red-100 hover:bg-red-200 text-red-600 rounded-lg delete-btn" data-item-type="servicio" data-item-id="${service.id}" title="ELIMINAR SERVICIO"><i data-lucide="trash-2" class="w-5 h-5"></i></button>
            </div>
        `;
        return card;
    }

    function renderServiceTableRow(service) {
        const row = document.createElement('tr');
        row.className = 'border-b border-gray-200 hover:bg-gray-50';
        row.dataset.servicioId = service.id;
        row.dataset.itemId = service.id;
        const visibilityText = service.show_on_web ?
            '<span class="bg-green-100 text-green-800 py-1 px-2 rounded-full text-xs font-semibold uppercase flex items-center justify-center gap-1"><i data-lucide="eye" class="w-4 h-4"></i> Visible</span>' :
            '<span class="bg-red-100 text-red-800 py-1 px-2 rounded-full text-xs font-semibold uppercase flex items-center justify-center gap-1"><i data-lucide="eye-off" class="w-4 h-4"></i> Oculto</span>';
        row.innerHTML = `
            <td class="py-4 px-6" data-label="NOMBRE">
                <div class="flex items-start flex-wrap"> <span class="text-blue-600 hover:text-blue-800 cursor-pointer font-semibold uppercase break-words">${service.nombre}</span> </div>
            </td>
            <td class="py-4 px-6 hidden md:table-cell" data-label="CATEGORÍA">${service.categoria || 'N/A'}</td>
            <td class="py-4 px-6 uppercase" data-label="PRECIO">$${parseFloat(service.precio).toFixed(2)}</td>
            <td class="py-4 px-6 text-center" data-label="VISIBILIDAD WEB">${visibilityText}</td>
            <td class="py-4 px-6 text-center" data-label="ACCIONES">
                <div class="flex flex-col sm:flex-row justify-center space-y-2 sm:space-y-0 sm:space-x-2"> <button class="text-blue-600 hover:text-blue-800 edit-servicio-btn" title="EDITAR SERVICIO" data-servicio-id="${service.id}"><i data-lucide="edit-2" class="w-5 h-5"></i></button>
                    <button class="text-red-600 hover:text-red-800 delete-btn" data-item-type="servicio" data-item-id="${service.id}" title="ELIMINAR SERVICIO"><i data-lucide="trash-2" class="w-5 h-5"></i></button>
                </div>
            </td>
        `;
        return row;
    }

    function renderServices(viewType = currentServiceView) {
        if (!serviceGridView || !serviceListView || !servicesTableBody || !serviceSearch || !serviceFilterCategory || !gridViewBtn || !listViewBtn) {
            showToast("[services.php] ERROR: Elementos DOM de Servicios no encontrados. Verifique los IDs de los elementos HTML.", 'error'); return;
        }
        const searchTerm = serviceSearch.value.toLowerCase();
        const selectedCategory = serviceFilterCategory.value;
        const allServicesArray = Object.values(serviciosActuales);

        // console.log("renderServices: serviciosActuales antes de filtrar:", allServicesArray); // Reemplazado

        const filteredServices = allServicesArray.filter(s => {
            const matchesSearch = searchTerm === '' || s.nombre.toLowerCase().includes(searchTerm) || (s.descripcion && s.descripcion.toLowerCase().includes(searchTerm));
            const matchesCategory = selectedCategory === 'all' || (s.categoria && s.categoria.toLowerCase().includes(selectedCategory.toLowerCase()));
            return matchesSearch && matchesCategory;
        });

        // console.log("renderServices: servicios filtrados:", filteredServices); // Reemplazado


        serviceGridView.innerHTML = '';
        servicesTableBody.innerHTML = '';

        if (filteredServices.length === 0) {
            const message = `<p class="text-gray-500 md:col-span-2 lg:col-span-3 text-center uppercase">NO SE ENCONTRARON SERVICIOS.</p>`;
            if (viewType === 'grid') {
                serviceGridView.innerHTML = message;
            } else {
                servicesTableBody.innerHTML = `<tr><td colspan="5" class="py-4 px-6 text-center text-gray-500 uppercase">NO SE ENCONTRARON SERVICIOS.</td></tr>`;
            }
        } else {
            if (viewType === 'grid') {
                filteredServices.forEach(service => serviceGridView.appendChild(renderServiceCard(service)));
            } else {
                filteredServices.forEach(service => servicesTableBody.appendChild(renderServiceTableRow(service)));
            }
        }

        if (viewType === 'grid') {
            serviceGridView.classList.remove('hidden');
            serviceListView.classList.add('hidden');
            gridViewBtn.classList.add('active');
            listViewBtn.classList.remove('active');
        } else {
            serviceGridView.classList.add('hidden');
            serviceListView.classList.remove('hidden');
            gridViewBtn.classList.remove('active');
            listViewBtn.classList.add('active');
        }

        setupDeleteListeners();
        setupEditListeners();
        lucide.createIcons();
    }
    
    // NOTA: Se ha eliminado la función 'cargarServiciosDesdeDB' que hacía un fetch,
    // ya que los datos se cargan directamente desde PHP al inicio de la página.
    // La variable 'serviciosActuales' ya contiene los datos.

    // Funciones de formateo de teléfono (se mantienen aunque no se usan directamente en esta página)
    function formatPhoneNumber(value) {
        if (!value) return '';
        const cleaned = ('' + value).replace(/\D/g, '');    
        const match = cleaned.match(/^(\d{3})(\d{3})(\d{4})$/);
        if (match) {
            return `(${match[1]}) ${match[2]}-${match[3]}`;
        }
        return cleaned;     
    }

    function applyInputPhoneFormat(inputElement) {
        if (!inputElement) return;
        inputElement.value = formatPhoneNumber(inputElement.value);
        inputElement.addEventListener('input', (e) => {
            let value = e.target.value.replace(/\D/g, '');     
            e.target.value = formatPhoneNumber(value);
        });
        inputElement.addEventListener('blur', (e) => {
            e.target.value = formatPhoneNumber(e.target.value);
        });
    }

    function applyPhoneFormatToInputs() {
        // No hay campos de teléfono en esta página de servicios, pero la función se mantiene.
        // Si se agregaran inputs con estos IDs en el futuro, se aplicarían.
        // applyInputPhoneFormat(document.getElementById('some-phone-input'));
    }


    // --- INICIALIZACIÓN DE LA PÁGINA ---
    // Esta es la primera y única función que se ejecuta al cargar el DOM.
    // El phpNotificationMessage se obtiene al principio de este mismo script.

    // Mostrar notificación PHP si existe
    const phpNotificationMessage = "<?php echo isset($_SESSION['notification']) ? addslashes($_SESSION['notification']['message']) : ''; ?>";
    const phpNotificationType = "<?php echo isset($_SESSION['notification']) ? addslashes($_SESSION['notification']['type']) : ''; ?>";
    <?php unset($_SESSION['notification']); // Limpiar la sesión después de leerla ?>

    if (phpNotificationMessage) {
        showToast(phpNotificationMessage, phpNotificationType);
    }

    // Eventos de la barra lateral (menú móvil)
    if (mobileMenuButton) mobileMenuButton.addEventListener('click', toggleSidebar);
    // Cierra los paneles y el overlay si se hace clic en el overlay de la barra lateral
    if (sidebarOverlay) sidebarOverlay.addEventListener('click', () => {
        toggleSidebar();
        closePanel('addServicioPanel'); // Cierra el panel de agregar
        closePanel('viewServicioPanel'); // Cierra el panel de ver/editar
        closePanel('manageCategoriesPanel'); // NUEVO: Cierra el panel de categorías
    });
    // Este listener en panelOverlay ahora es crucial para cerrar los paneles laterales
    panelOverlay.addEventListener('click', () => {    
        closePanel('addServicioPanel');
        closePanel('viewServicioPanel');
        closePanel('manageCategoriesPanel'); // NUEVO: Cierra el panel de categorías
    });

    // --- Carga inicial de categorías y renderizado ---
    fetchCategories(); // Carga las categorías al inicio
    populateCategoryFilter(); // Popula el filtro (lo hará de nuevo después de fetchCategories)
    populateCategoryDropdowns(); // Popula los dropdowns (lo hará de nuevo después de fetchCategories)
    updateTotalServicesDisplay(); // Actualiza el conteo total de servicios
    monthlySoldServicesCountElement.textContent = monthlySoldServicesCount; // Actualiza el contador de servicios vendidos al mes


    // Renderizar servicios en la vista inicial (grid por defecto)
    renderServices(currentServiceView);
    // console.log("[services.php] Servicios renderizados."); // Reemplazado

    if (serviceSearch) serviceSearch.addEventListener('input', () => renderServices(currentServiceView));
    if (serviceFilterCategory) serviceFilterCategory.addEventListener('change', () => renderServices(currentServiceView));

    if(downloadServiciosBtn) {
        downloadServiciosBtn.addEventListener('click', () => {
            // Los datos ya están en serviciosActuales directamente desde PHP, listos para descargar
            descargarComoCSV(serviciosActuales, 'catalogo_servicios.csv');
        });
    }
    
    // Lógica para los formularios (sin cambios significativos en el manejo de eventos)
    if (confirmDeleteButton) {
        confirmDeleteButton.addEventListener('click', function() {
            const itemId = this.dataset.itemId;
            const itemType = this.dataset.itemType;
            if (itemType === 'servicio') {
                const dbId = serviciosActuales[itemId] ? serviciosActuales[itemId].db_id : null;
                if (dbId) {
                    fetch('db/services-delete.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify({ id: dbId })
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            closeModal('confirmDeleteModal');
                            showToast('Servicio eliminado con éxito.', 'success');
                            window.location.reload(); // Recarga toda la página para obtener los datos actualizados
                        } else {
                            showToast('Error al eliminar servicio: ' + data.message, 'error');
                        }
                    })
                    .catch(error => showToast('Error de conexión al eliminar el servicio.', 'error'));
                }
            }
        });
    }

    if (gridViewBtn) {
        gridViewBtn.addEventListener('click', () => {
            currentServiceView = 'grid';
            renderServices('grid');
        });
    }
    if (listViewBtn) {
        listViewBtn.addEventListener('click', () => {
            currentServiceView = 'list';
            renderServices('list');
        });
    }

    if (addServicioForm) {
        addServicioForm.addEventListener('submit', async function(e) { // Marcado como async
            e.preventDefault();
            const serviceData = {
                name: document.getElementById('servicio-nombre').value,
                price: parseFloat(document.getElementById('servicio-precio').value),
                category: addServiceCategorySelect.value, // Obtener valor del SELECT
                description: document.getElementById('servicio-descripcion').value,
                show_on_web: servicioMostrarWebCheckbox.checked
            };

            try { // Agregado try-catch
                const response = await fetch('db/services-create.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(serviceData)
                });
                const data = await response.json();
                if (data.success) {
                    closePanel('addServicioPanel');
                    addServicioForm.reset();
                    showToast('Servicio agregado con éxito.', 'success');
                    window.location.reload();
                } else {
                    showToast('Error al agregar servicio: ' + (data.message || 'Error desconocido.'), 'error');
                }
            } catch (error) {
                showToast('Error de conexión al agregar el servicio.', 'error');
                console.error('Create service error:', error);
            }
        });
    }

    if (viewServicioForm) {
        viewServicioForm.addEventListener('submit', async function(e) { // Marcado como async
            e.preventDefault();
            const serviceData = {
                id: document.getElementById('view-servicio-id').value,
                name: document.getElementById('view-servicio-nombre').value,
                price: parseFloat(document.getElementById('view-servicio-precio').value),
                category: viewServiceCategorySelect.value, // Obtener valor del SELECT
                description: document.getElementById('view-servicio-descripcion').value,
                show_on_web: viewServicioMostrarWebCheckbox.checked
            };
            
            try { // Agregado try-catch
                const response = await fetch('db/services-update.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify(serviceData)
                });
                const data = await response.json();
                if (data.success) {
                    closePanel('viewServicioPanel');
                    showToast('Cambios guardados con éxito.', 'success');
                    window.location.reload();
                } else {
                    showToast('Error al guardar cambios: ' + (data.message || 'Error desconocido.'), 'error');
                }
            } catch (error) {
                showToast('Error de conexión al guardar los cambios.', 'error');
                console.error('Update service error:', error);
            }
        });
    }

    // --- NUEVO: Lógica para el panel de administración de categorías ---
    if (addCategoryForm) {
        addCategoryForm.addEventListener('submit', async function(e) { // Marcado como async
            e.preventDefault();
            const newCategoryName = newCategoryNameInput.value.trim();
            if (newCategoryName) {
                showToast(`Añadiendo categoría...`, 'info');
                try {
                    const response = await fetch('db/services-categories-create.php', { // Usa el nuevo nombre de archivo
                        method: 'POST',
                        headers: {'Content-Type': 'application/json'},
                        body: JSON.stringify({ name: newCategoryName })
                    });
                    const data = await response.json();
                    if (data.success) {
                        showToast('Categoría añadida con éxito.', 'success');
                        newCategoryNameInput.value = ''; // Limpiar campo
                        await fetchCategories(); // Recargar categorías del servidor
                    } else {
                        showToast('Error al añadir categoría: ' + (data.message || 'Error desconocido.'), 'error');
                    }
                } catch (error) {
                    showToast('Error de conexión al añadir la categoría.', 'error');
                    console.error('Create category error:', error);
                }
            } else {
                showToast('El nombre de la categoría no puede estar vacío.', 'error');
            }
        });
    }
});
</script>
</body>
</html>